<?php
require_once 'auth.php';

$waitlist_file  = __DIR__ . '/data/waitlist.csv';
$inventory_file = __DIR__ . '/data/inventory.csv';
$settings_file  = __DIR__ . '/data/org_settings.json';

// ---- Handle org settings save (editor + admin only) ----
$settings_msg = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_settings') {
    require_editor();
    verify_csrf();
    $fields = ['org_name','drop_address_line1','drop_address_line2','drop_city_state_zip',
               'drop_hours','drop_notes','contact_name','contact_email','contact_phone',
               'tax_contact_name','tax_contact_email','tax_contact_phone','tax_id',
               'welcome_intro','welcome_accept','welcome_dropoff_note',
               'welcome_requests','welcome_tagline'];
    $save = [];
    foreach ($fields as $f) $save[$f] = trim($_POST[$f] ?? '');
    file_put_contents($settings_file, json_encode($save, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    $settings_msg = 'Settings saved.';
}

// Load org settings for display in form
$org_defaults = [
    'org_name'             => 'Laptop Ministry',
    'drop_address_line1'   => '', 'drop_address_line2' => '', 'drop_city_state_zip' => '',
    'drop_hours'           => '', 'drop_notes'         => '',
    'contact_name'         => '', 'contact_email'      => '', 'contact_phone'       => '',
    'tax_contact_name'     => '', 'tax_contact_email'  => '', 'tax_contact_phone'   => '',
    'tax_id'               => '',
    'welcome_intro'        => 'We collect older laptops, securely wipe them, refurbish them, and install an easy-to-use Linux operating system—then give them at no cost to Chicago neighbors who need one most.',
    'welcome_accept'       => "Laptops in working condition\nManufactured 2012 or newer",
    'welcome_dropoff_note' => 'Reception desk (1st floor) of the Inspiration Corporation building',
    'welcome_requests'     => "Messages about availability are usually answered on Sundays. Pickups (and special drop-off arrangements) can be coordinated once we confirm we have a suitable device for you. Because demand far outpaces our current capacity, we prioritize people with the greatest need.\n\nWe\xe2\x80\x99d love to see sister ministries like this one start in other neighborhoods and cities!\n\nThis effort became an official ministry of Peoples Church of Chicago in April 2023. For the latest details, to get in touch, or to learn how you can help, visit laptopministry.org",
    'welcome_tagline'      => "Thank you for helping reduce e-waste and close the digital divide\xe2\x80\x94one renewed laptop at a time!",
];
$org_raw = file_exists($settings_file) ? json_decode(file_get_contents($settings_file), true) : [];
$org = array_merge($org_defaults, is_array($org_raw) ? $org_raw : []);

// Waitlist statistics
function getWaitlistStats($file) {
    $stats = [
        'total_waitlist' => 0,
        'laptops_provided' => 0,
        'avg_rate' => 0,
        'first_pickup' => 'N/A',
        'last_pickup' => 'N/A',
        'total_entries' => 0
    ];
    if (!file_exists($file)) return $stats;

    $fh = fopen($file, 'r');
    $headers = fgetcsv($fh);
    if (!$headers) { fclose($fh); return $stats; }

    $pickup_dates = [];
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) < 13) continue;
        $stats['total_entries']++;
        $picked_up = trim($row[11] ?? '');
        $pickup_date_str = trim($row[12] ?? '');

        if ($picked_up === 'N') $stats['total_waitlist']++;
        if ($picked_up === 'Y' && !empty($pickup_date_str)) {
            $stats['laptops_provided']++;
            $date = false;
            foreach (['m/d/y', 'Y-m-d', 'Ymd'] as $fmt) {
                $d = DateTime::createFromFormat($fmt, $pickup_date_str);
                if ($d && $d->format($fmt) === $pickup_date_str) { $date = $d; break; }
            }
            if ($date) $pickup_dates[] = $date;
        }
    }
    fclose($fh);

    if (!empty($pickup_dates)) {
        $earliest = min($pickup_dates);
        $latest = max($pickup_dates);
        $days = $latest->diff($earliest)->days;
        if ($days > 0) {
            $stats['avg_rate'] = round($stats['laptops_provided'] / ($days / 30.44), 2);
        }
        $stats['first_pickup'] = $earliest->format('m/d/Y');
        $stats['last_pickup'] = $latest->format('m/d/Y');
    }
    return $stats;
}

// Inventory statistics
function getInventoryStats($file) {
    $stats = [
        'total_items'           => 0,
        'total_value'           => 0,
        'active_items'          => 0,
        'computers_fulfillable' => 0,
        'computers_repair'      => 0,
        'computers_partsonly'   => 0,
        'computers_vintage'     => 0,
        'active_parts'          => 0,
        'discarded_items'       => 0
    ];
    if (!file_exists($file)) return $stats;

    $fh = fopen($file, 'r');
    $headers = fgetcsv($fh);
    if (!$headers) { fclose($fh); return $stats; }

    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) < 9) continue;
        $stats['total_items']++;
        $value       = floatval($row[6] ?? 0);
        $discarded   = trim($row[8] ?? '');
        $item_type   = trim($row[10] ?? '');
        $item_status = trim($row[11] ?? '');
        $stats['total_value'] += $value;
        if ($discarded === 'Yes') {
            $stats['discarded_items']++;
        } else {
            $stats['active_items']++;
            if ($item_type === 'Part') {
                $stats['active_parts']++;
            } elseif ($item_status === 'Repair') {
                $stats['computers_repair']++;
            } elseif ($item_status === 'Parts') {
                $stats['computers_partsonly']++;
            } elseif ($item_status === 'Vintage') {
                $stats['computers_vintage']++;
            } else {
                $stats['computers_fulfillable']++;
            }
        }
    }
    fclose($fh);
    return $stats;
}

$wStats = getWaitlistStats($waitlist_file);
$iStats = getInventoryStats($inventory_file);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Laptop Ministry Accounting</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
</head>
<body>
    <div class="admin-wrapper">
        <?php include '_sidebar.php'; ?>
        <main class="content">
            <div class="window">
                <div class="window-title">Dashboard</div>
                <div class="window-content">
                    <h2>Waitlist</h2>
                    <div class="stats-grid">
                        <div class="stat-box">
                            <div class="stat-value"><?= $wStats['total_waitlist'] ?></div>
                            <div class="stat-label">People Waiting</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $wStats['laptops_provided'] ?></div>
                            <div class="stat-label">Laptops Provided</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $wStats['avg_rate'] ?></div>
                            <div class="stat-label">Laptops / Month</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= htmlspecialchars($wStats['first_pickup']) ?></div>
                            <div class="stat-label">First Pickup</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= htmlspecialchars($wStats['last_pickup']) ?></div>
                            <div class="stat-label">Last Pickup</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $wStats['total_entries'] ?></div>
                            <div class="stat-label">Total Entries</div>
                        </div>
                    </div>

                    <h2>Inventory</h2>
                    <div class="stats-grid">
                        <div class="stat-box">
                            <div class="stat-value"><?= $iStats['total_items'] ?></div>
                            <div class="stat-label">Total Items</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value">$<?= number_format($iStats['total_value'], 2) ?></div>
                            <div class="stat-label">Total Value</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $iStats['active_items'] ?></div>
                            <div class="stat-label">Active Items</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $iStats['computers_fulfillable'] ?></div>
                            <div class="stat-label">Fulfillable Computers</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $iStats['computers_repair'] ?></div>
                            <div class="stat-label">Needs Repair</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $iStats['computers_partsonly'] ?></div>
                            <div class="stat-label">Used for Parts</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $iStats['computers_vintage'] ?></div>
                            <div class="stat-label">Vintage / Special</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $iStats['active_parts'] ?></div>
                            <div class="stat-label">Parts / Accessories</div>
                        </div>
                        <div class="stat-box">
                            <div class="stat-value"><?= $iStats['discarded_items'] ?></div>
                            <div class="stat-label">Discarded</div>
                        </div>
                    </div>

                    <h2>Quick Links</h2>
                    <div class="quick-links">
                        <a href="waitlist.php">Manage Waitlist</a>
                        <a href="inventory.php">Manage Inventory</a>
                        <a href="import.php">Import Data</a>
                        <a href="export.php">Export Data</a>
                    </div>

                    <br>
                    <p>Logged in as <strong><?= htmlspecialchars($_SESSION['username']) ?></strong> (<?= htmlspecialchars($_SESSION['role']) ?>)</p>
                    <p><em>Laptop Ministry Accounting System &mdash; <?= date('Y') ?> Peoples Church of Chicago</em></p>
                </div>
            </div>

            <?php if (can_edit()): ?>
            <!-- ---- Org Settings ---- -->
            <div class="window" style="margin-top:14px;">
                <div class="window-title">&#9881; Public Page Settings &mdash; Donation &amp; Contact Info</div>
                <div class="window-content">
                    <p style="font-size:12px;margin:0 0 10px;color:#444;">
                        These details appear on the public home page in the <strong>Donate a Laptop</strong> panel.
                        Leave fields blank to hide that section from visitors.
                    </p>
                    <?php if ($settings_msg): ?>
                        <p class="success"><?= htmlspecialchars($settings_msg) ?></p>
                    <?php endif; ?>
                    <form method="post">
                        <input type="hidden" name="action" value="save_settings">
                        <input type="hidden" name="csrf_token" value="<?= csrf_token() ?>">

                        <fieldset>
                            <legend>Organization</legend>
                            <div class="form-row">
                                <label>Organization Name</label>
                                <input type="text" name="org_name" value="<?= htmlspecialchars($org['org_name']) ?>">
                            </div>
                        </fieldset>

                        <fieldset style="margin-top:10px;">
                            <legend>Drop-off / Donation Address</legend>
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>Address Line 1</label>
                                    <input type="text" name="drop_address_line1" value="<?= htmlspecialchars($org['drop_address_line1']) ?>" placeholder="Street address">
                                </div>
                                <div class="form-row">
                                    <label>Address Line 2</label>
                                    <input type="text" name="drop_address_line2" value="<?= htmlspecialchars($org['drop_address_line2']) ?>" placeholder="Suite, room, c/o...">
                                </div>
                                <div class="form-row">
                                    <label>City, State, ZIP</label>
                                    <input type="text" name="drop_city_state_zip" value="<?= htmlspecialchars($org['drop_city_state_zip']) ?>" placeholder="Chicago, IL 60601">
                                </div>
                            </div>
                            <div class="form-row">
                                <label>Drop-off Hours</label>
                                <input type="text" name="drop_hours" value="<?= htmlspecialchars($org['drop_hours']) ?>" placeholder="e.g. Mon–Fri 9am–5pm, or By appointment">
                            </div>
                            <div class="form-row">
                                <label>Additional Drop-off Notes</label>
                                <textarea name="drop_notes" placeholder="Parking info, entrance instructions, etc."><?= htmlspecialchars($org['drop_notes']) ?></textarea>
                            </div>
                        </fieldset>

                        <fieldset style="margin-top:10px;">
                            <legend>General Donation Contact</legend>
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>Name / Title</label>
                                    <input type="text" name="contact_name" value="<?= htmlspecialchars($org['contact_name']) ?>" placeholder="e.g. Donations Coordinator">
                                </div>
                                <div class="form-row">
                                    <label>Email</label>
                                    <input type="email" name="contact_email" value="<?= htmlspecialchars($org['contact_email']) ?>">
                                </div>
                                <div class="form-row">
                                    <label>Phone</label>
                                    <input type="tel" name="contact_phone" value="<?= htmlspecialchars($org['contact_phone']) ?>">
                                </div>
                            </div>
                        </fieldset>

                        <fieldset style="margin-top:10px;">
                            <legend>Tax-Exempt / Donation Receipt Contact</legend>
                            <p style="font-size:11px;margin:0 0 8px;color:#555;">
                                Shown to donors who need a tax-exempt letter. After a donation is logged, staff can issue
                                the letter from the Inventory record.
                            </p>
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>EIN / Tax-Exempt ID</label>
                                    <input type="text" name="tax_id" value="<?= htmlspecialchars($org['tax_id']) ?>" placeholder="xx-xxxxxxx">
                                </div>
                                <div class="form-row">
                                    <label>Name / Title</label>
                                    <input type="text" name="tax_contact_name" value="<?= htmlspecialchars($org['tax_contact_name']) ?>" placeholder="e.g. Church Administrator">
                                </div>
                                <div class="form-row">
                                    <label>Email</label>
                                    <input type="email" name="tax_contact_email" value="<?= htmlspecialchars($org['tax_contact_email']) ?>">
                                </div>
                                <div class="form-row">
                                    <label>Phone</label>
                                    <input type="tel" name="tax_contact_phone" value="<?= htmlspecialchars($org['tax_contact_phone']) ?>">
                                </div>
                            </div>
                        </fieldset>

                        <fieldset style="margin-top:10px;">
                            <legend>Welcome Modal &mdash; Public Home Page Pop-up</legend>
                            <p style="font-size:11px;margin:0 0 8px;color:#555;">
                                Shown once to new visitors. Leave a field blank to hide that section.
                                The drop-off address shown in the modal uses the fields above.
                                For the <em>What we accept</em> list, put one item per line.
                            </p>
                            <div class="form-row">
                                <label>Opening paragraph</label>
                                <textarea name="welcome_intro" rows="3"><?= htmlspecialchars($org['welcome_intro']) ?></textarea>
                            </div>
                            <div class="form-row">
                                <label>What we accept <em style="font-weight:normal;">(one item per line &rarr; bullet list)</em></label>
                                <textarea name="welcome_accept" rows="3"><?= htmlspecialchars($org['welcome_accept']) ?></textarea>
                            </div>
                            <div class="form-row">
                                <label>Drop-off location label</label>
                                <input type="text" name="welcome_dropoff_note" value="<?= htmlspecialchars($org['welcome_dropoff_note']) ?>" placeholder="e.g. Reception desk (1st floor) of the Inspiration Corporation building">
                            </div>
                            <div class="form-row">
                                <label>Laptop request &amp; extra notes <em style="font-weight:normal;">(blank lines become paragraph breaks)</em></label>
                                <textarea name="welcome_requests" rows="6"><?= htmlspecialchars($org['welcome_requests']) ?></textarea>
                            </div>
                            <div class="form-row">
                                <label>Closing / thank-you line</label>
                                <input type="text" name="welcome_tagline" value="<?= htmlspecialchars($org['welcome_tagline']) ?>">
                            </div>
                        </fieldset>

                        <button type="submit" style="margin-top:10px;">Save Settings</button>
                    </form>
                </div>
            </div>
            <?php endif; ?>

        </main>
    </div>
</body>
</html>
