<?php
require_once 'auth.php';
require_editor();

$waitlist_file = __DIR__ . '/data/waitlist.csv';
$inventory_file = __DIR__ . '/data/inventory.csv';

$waitlist_headers = ['number','first_name','last_name','contact','store_date','request_date','priority','computer_ready','inventory_id','computer_desc','computer_os','picked_up','pickup_date','need_desc','notes'];
$inventory_headers = ['id','entry_date','date_received','donor_name','donor_contact','tax_letter','item_value','item_description','discarded','notes','item_type','item_status'];
$applicant_headers = ['first_name','last_name','contact','request_date','need_desc','notes'];

$message = '';
$msg_type = '';
$preview_rows = [];
$preview_errors = [];
$preview_target = '';
$preview_mode = '';
$csv_data_json = '';

// Step 1: Upload and validate
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    verify_csrf();

    if ($_POST['action'] === 'upload') {
        $target = $_POST['target'] ?? '';
        $mode = $_POST['import_mode'] ?? 'standard';

        if (!isset($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
            $message = 'File upload failed. Please select a valid CSV file.';
            $msg_type = 'error';
        } elseif ($_FILES['csv_file']['size'] > 2 * 1024 * 1024) {
            $message = 'File too large. Maximum size is 2MB.';
            $msg_type = 'error';
        } else {
            $ext = strtolower(pathinfo($_FILES['csv_file']['name'], PATHINFO_EXTENSION));
            if ($ext !== 'csv') {
                $message = 'Only .csv files are accepted.';
                $msg_type = 'error';
            } else {
                $tmp = $_FILES['csv_file']['tmp_name'];
                $fh = fopen($tmp, 'r');
                $file_headers = fgetcsv($fh);

                if (!$file_headers) {
                    $message = 'Could not read CSV headers. File may be empty or malformed.';
                    $msg_type = 'error';
                } else {
                    $file_headers = array_map('trim', $file_headers);
                    $file_headers = array_map('strtolower', $file_headers);

                    // Determine expected headers
                    if ($mode === 'applicant') {
                        $expected = $applicant_headers;
                        $preview_target = 'waitlist';
                    } elseif ($target === 'waitlist') {
                        $expected = $waitlist_headers;
                        $preview_target = 'waitlist';
                    } elseif ($target === 'inventory') {
                        $expected = $inventory_headers;
                        $preview_target = 'inventory';
                    } else {
                        $message = 'Please select a target (Waitlist or Inventory).';
                        $msg_type = 'error';
                        $expected = [];
                    }

                    if (!empty($expected)) {
                        // Check header match
                        $header_match = true;
                        $missing = array_diff($expected, $file_headers);
                        $extra = array_diff($file_headers, $expected);

                        if (!empty($missing)) {
                            $message = 'Missing columns: ' . implode(', ', $missing) . '.';
                            $msg_type = 'error';
                            $header_match = false;
                        }

                        if ($header_match) {
                            // Parse and validate rows
                            $row_num = 1;
                            $parsed_rows = [];
                            while (($row = fgetcsv($fh)) !== false) {
                                $row_num++;
                                $row_data = [];
                                $row_errors = [];

                                if (count($row) < count($expected)) {
                                    $row = array_pad($row, count($expected), '');
                                    $row_errors[] = 'Row has fewer columns than expected (padded with blanks)';
                                }
                                if (count($row) > count($expected)) {
                                    $row = array_slice($row, 0, count($expected));
                                    $row_errors[] = 'Row has extra columns (truncated)';
                                }

                                // Map by header position
                                $mapped = [];
                                foreach ($expected as $i => $col) {
                                    $file_col_idx = array_search($col, $file_headers);
                                    $mapped[$col] = ($file_col_idx !== false && isset($row[$file_col_idx])) ? trim($row[$file_col_idx]) : '';
                                }

                                // Validate based on target
                                if ($preview_target === 'waitlist' && $mode !== 'applicant') {
                                    if (empty($mapped['first_name'])) $row_errors[] = 'first_name is empty';
                                    if (empty($mapped['last_name'])) $row_errors[] = 'last_name is empty';
                                    if (!empty($mapped['request_date']) && !strtotime($mapped['request_date'])) $row_errors[] = 'request_date is not a valid date';
                                    if (!empty($mapped['pickup_date']) && !strtotime($mapped['pickup_date'])) $row_errors[] = 'pickup_date is not a valid date';
                                    if (!empty($mapped['priority']) && !in_array($mapped['priority'], ['Y','N',''])) $row_errors[] = 'priority must be Y or N';
                                    if (!empty($mapped['computer_ready']) && !in_array($mapped['computer_ready'], ['Y','N',''])) $row_errors[] = 'computer_ready must be Y or N';
                                    if (!empty($mapped['picked_up']) && !in_array($mapped['picked_up'], ['Y','N',''])) $row_errors[] = 'picked_up must be Y or N';
                                } elseif ($mode === 'applicant') {
                                    if (empty($mapped['first_name']) && empty($mapped['last_name'])) $row_errors[] = 'Name is empty';
                                    if (empty($mapped['contact'])) $row_errors[] = 'contact is empty';
                                } elseif ($preview_target === 'inventory') {
                                    if (empty($mapped['donor_name'])) $row_errors[] = 'donor_name is empty';
                                    if (empty($mapped['item_description'])) $row_errors[] = 'item_description is empty';
                                    if (!empty($mapped['date_received']) && !strtotime($mapped['date_received'])) $row_errors[] = 'date_received is not a valid date';
                                    if (!empty($mapped['item_value']) && !is_numeric($mapped['item_value'])) $row_errors[] = 'item_value must be a number';
                                    if (!empty($mapped['tax_letter']) && !in_array($mapped['tax_letter'], ['Yes','No',''])) $row_errors[] = 'tax_letter must be Yes or No';
                                    if (!empty($mapped['discarded']) && !in_array($mapped['discarded'], ['Yes','No',''])) $row_errors[] = 'discarded must be Yes or No';
                                    if (!empty($mapped['item_type']) && !in_array($mapped['item_type'], ['Computer','Part',''])) $row_errors[] = 'item_type must be Computer or Part';
                                    if (!empty($mapped['item_status']) && !in_array($mapped['item_status'], ['Ready','Repair','Parts',''])) $row_errors[] = 'item_status must be Ready, Repair, or Parts';
                                }

                                $parsed_rows[] = $mapped;
                                $preview_errors[] = $row_errors;
                            }

                            $preview_rows = $parsed_rows;
                            $preview_mode = $mode;
                            $csv_data_json = base64_encode(json_encode($parsed_rows));
                        }
                    }
                }
                fclose($fh);
            }
        }
    }

    // Step 2: Confirm import
    if ($_POST['action'] === 'confirm_import') {
        $target = $_POST['confirm_target'] ?? '';
        $mode = $_POST['confirm_mode'] ?? 'standard';
        $skip_invalid = isset($_POST['skip_invalid']);
        $raw = base64_decode($_POST['csv_data'] ?? '');
        $rows = json_decode($raw, true);

        if (!$rows || !is_array($rows)) {
            $message = 'Import data corrupted. Please re-upload.';
            $msg_type = 'error';
        } else {
            $errors_json = json_decode(base64_decode($_POST['errors_data'] ?? ''), true) ?: [];
            $imported = 0;
            $skipped = 0;

            if ($target === 'waitlist') {
                // Read existing waitlist to get next number
                $existing = [];
                if (file_exists($waitlist_file)) {
                    $efh = fopen($waitlist_file, 'r');
                    fgetcsv($efh); // skip headers
                    while (($er = fgetcsv($efh)) !== false) $existing[] = $er;
                    fclose($efh);
                }
                $next_num = 1;
                foreach ($existing as $e) { $n = intval($e[0]); if ($n >= $next_num) $next_num = $n + 1; }

                $fh = fopen($waitlist_file, 'a');
                if (!$fh) { $message = 'Cannot write to waitlist file.'; $msg_type = 'error'; }
                else {
                    flock($fh, LOCK_EX);
                    // If file was empty, write headers
                    if (empty($existing)) {
                        // Reopen in write mode
                        flock($fh, LOCK_UN);
                        fclose($fh);
                        $fh = fopen($waitlist_file, 'w');
                        flock($fh, LOCK_EX);
                        fputcsv($fh, $waitlist_headers);
                    }

                    foreach ($rows as $i => $row) {
                        $has_errors = !empty($errors_json[$i]);
                        if ($has_errors && $skip_invalid) { $skipped++; continue; }
                        if ($has_errors && !$skip_invalid) { $skipped++; continue; }

                        if ($mode === 'applicant') {
                            // Map applicant fields to waitlist
                            $entry = [
                                $next_num++,
                                $row['first_name'] ?? '',
                                $row['last_name'] ?? '',
                                $row['contact'] ?? '',
                                date('Ymd'),
                                $row['request_date'] ?? date('Y-m-d'),
                                'N', // priority
                                'N', // computer_ready
                                '',  // inventory_id
                                '',  // computer_desc
                                '',  // computer_os
                                'N', // picked_up
                                '',  // pickup_date
                                $row['need_desc'] ?? '',
                                $row['notes'] ?? ''
                            ];
                        } else {
                            $entry = [
                                $row['number'] ?? $next_num++,
                                $row['first_name'] ?? '',
                                $row['last_name'] ?? '',
                                $row['contact'] ?? '',
                                $row['store_date'] ?? date('Ymd'),
                                $row['request_date'] ?? '',
                                $row['priority'] ?? 'N',
                                $row['computer_ready'] ?? 'N',
                                $row['inventory_id'] ?? '',
                                $row['computer_desc'] ?? '',
                                $row['computer_os'] ?? '',
                                $row['picked_up'] ?? 'N',
                                $row['pickup_date'] ?? '',
                                $row['need_desc'] ?? '',
                                $row['notes'] ?? ''
                            ];
                        }
                        fputcsv($fh, $entry);
                        $imported++;
                    }
                    flock($fh, LOCK_UN);
                    fclose($fh);
                    $message = "Imported $imported entries to waitlist.";
                    if ($skipped > 0) $message .= " Skipped $skipped invalid rows.";
                    $msg_type = 'success';
                }
            } elseif ($target === 'inventory') {
                $existing = [];
                if (file_exists($inventory_file)) {
                    $efh = fopen($inventory_file, 'r');
                    fgetcsv($efh);
                    while (($er = fgetcsv($efh)) !== false) $existing[] = $er;
                    fclose($efh);
                }
                $existing_ids = array_column($existing, 0);
                $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';

                $fh = fopen($inventory_file, 'a');
                if (!$fh) { $message = 'Cannot write to inventory file.'; $msg_type = 'error'; }
                else {
                    flock($fh, LOCK_EX);
                    if (empty($existing)) {
                        flock($fh, LOCK_UN);
                        fclose($fh);
                        $fh = fopen($inventory_file, 'w');
                        flock($fh, LOCK_EX);
                        fputcsv($fh, $inventory_headers);
                    }

                    foreach ($rows as $i => $row) {
                        $has_errors = !empty($errors_json[$i]);
                        if ($has_errors && $skip_invalid) { $skipped++; continue; }
                        if ($has_errors && !$skip_invalid) { $skipped++; continue; }

                        // Generate new ID if not provided or duplicate
                        $id = $row['id'] ?? '';
                        if (empty($id) || in_array($id, $existing_ids)) {
                            do {
                                $id = '';
                                for ($j = 0; $j < 4; $j++) $id .= $chars[random_int(0, strlen($chars)-1)];
                            } while (in_array($id, $existing_ids));
                        }
                        $existing_ids[] = $id;

                        $entry = [
                            $id,
                            $row['entry_date'] ?? date('Y-m-d H:i:s'),
                            $row['date_received'] ?? '',
                            $row['donor_name'] ?? '',
                            $row['donor_contact'] ?? '',
                            $row['tax_letter'] ?? 'No',
                            $row['item_value'] ?? '0',
                            $row['item_description'] ?? '',
                            $row['discarded'] ?? 'No',
                            $row['notes'] ?? '',
                            !empty($row['item_type'])   ? $row['item_type']   : 'Computer',
                            !empty($row['item_status']) ? $row['item_status'] : 'Ready'
                        ];
                        fputcsv($fh, $entry);
                        $imported++;
                    }
                    flock($fh, LOCK_UN);
                    fclose($fh);
                    $message = "Imported $imported items to inventory.";
                    if ($skipped > 0) $message .= " Skipped $skipped invalid rows.";
                    $msg_type = 'success';
                }
            }
        }
    }
}

$token = csrf_token();
$total_errors = 0;
foreach ($preview_errors as $errs) { if (!empty($errs)) $total_errors++; }
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Import - Laptop Ministry Accounting</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
</head>
<body>
    <div class="admin-wrapper">
        <?php include '_sidebar.php'; ?>
        <main class="content">
            <div class="window">
                <div class="window-title">Import Data</div>
                <div class="window-content">

                    <?php if ($message): ?>
                        <p class="<?= $msg_type ?>"><?= htmlspecialchars($message) ?></p>
                    <?php endif; ?>

                    <?php if (empty($preview_rows)): ?>
                    <!-- Upload Form -->
                    <p>Upload a CSV file to import into the waitlist or inventory. The file will be validated before import.</p>

                    <form method="post" enctype="multipart/form-data">
                        <input type="hidden" name="action" value="upload">
                        <input type="hidden" name="csrf_token" value="<?= $token ?>">

                        <fieldset>
                            <legend>Import Type</legend>
                            <div class="form-row">
                                <label>
                                    <input type="radio" name="import_mode" value="standard" checked onchange="toggleTarget(true)">
                                    Standard CSV Import
                                </label>
                                <label>
                                    <input type="radio" name="import_mode" value="applicant" onchange="toggleTarget(false)">
                                    Import from Applicant Survey Export
                                </label>
                            </div>
                        </fieldset>

                        <fieldset id="target-section">
                            <legend>Target</legend>
                            <div class="form-row">
                                <label>
                                    <input type="radio" name="target" value="waitlist" checked> Waitlist
                                </label>
                                <label>
                                    <input type="radio" name="target" value="inventory"> Inventory
                                </label>
                            </div>
                        </fieldset>

                        <fieldset>
                            <legend>CSV File</legend>
                            <div class="upload-area">
                                <p>Select a .csv file (max 2MB)</p>
                                <input type="file" name="csv_file" accept=".csv" required>
                            </div>
                        </fieldset>

                        <br>
                        <button type="submit">Upload &amp; Validate</button>
                    </form>

                    <br>
                    <fieldset>
                        <legend>Expected CSV Formats</legend>
                        <p><strong>Waitlist:</strong><br>
                        <code>number, first_name, last_name, contact, store_date, request_date, priority, computer_ready, inventory_id, computer_desc, computer_os, picked_up, pickup_date, need_desc, notes</code></p>
                        <p><strong>Inventory:</strong><br>
                        <code>id, entry_date, date_received, donor_name, donor_contact, tax_letter, item_value, item_description, discarded, notes, item_type, item_status</code><br>
                        <small>item_type: <em>Computer</em> (default) or <em>Part</em> &mdash; Parts do not count toward fulfillment.<br>
                        item_status: <em>Ready</em> (default), <em>Repair</em> (needs repair), or <em>Parts</em> (used for parts) &mdash; Repair and Parts do not count toward fulfillable supply.</small></p>
                        <p><strong>Applicant Survey Export:</strong><br>
                        <code>first_name, last_name, contact, request_date, need_desc, notes</code></p>
                    </fieldset>

                    <script>
                    function toggleTarget(show) {
                        document.getElementById('target-section').style.display = show ? 'block' : 'none';
                    }
                    </script>

                    <?php else: ?>
                    <!-- Preview Table -->
                    <h3>Import Preview &mdash; <?= htmlspecialchars(ucfirst($preview_target)) ?></h3>
                    <p>
                        <?= count($preview_rows) ?> rows found.
                        <?php if ($total_errors > 0): ?>
                            <span class="warning"><?= $total_errors ?> row(s) have validation issues.</span>
                        <?php else: ?>
                            <span class="success">All rows passed validation.</span>
                        <?php endif; ?>
                    </p>

                    <div class="table-scroll">
                        <table class="data-table preview-table">
                            <thead>
                                <tr>
                                    <th>Row</th>
                                    <?php
                                    $display_headers = ($preview_mode === 'applicant') ? $applicant_headers :
                                        ($preview_target === 'waitlist' ? $waitlist_headers : $inventory_headers);
                                    foreach ($display_headers as $h): ?>
                                        <th><?= htmlspecialchars($h) ?></th>
                                    <?php endforeach; ?>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($preview_rows as $i => $row): ?>
                                    <?php $has_err = !empty($preview_errors[$i]); ?>
                                    <tr class="<?= $has_err ? 'row-invalid' : 'row-valid' ?>">
                                        <td><?= $i + 1 ?></td>
                                        <?php foreach ($display_headers as $h): ?>
                                            <td><?= htmlspecialchars($row[$h] ?? '') ?></td>
                                        <?php endforeach; ?>
                                        <td>
                                            <?php if ($has_err): ?>
                                                <span class="error"><?= htmlspecialchars(implode('; ', $preview_errors[$i])) ?></span>
                                            <?php else: ?>
                                                <span class="success">OK</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <br>
                    <form method="post">
                        <input type="hidden" name="action" value="confirm_import">
                        <input type="hidden" name="csrf_token" value="<?= $token ?>">
                        <input type="hidden" name="confirm_target" value="<?= htmlspecialchars($preview_target) ?>">
                        <input type="hidden" name="confirm_mode" value="<?= htmlspecialchars($preview_mode) ?>">
                        <input type="hidden" name="csv_data" value="<?= $csv_data_json ?>">
                        <input type="hidden" name="errors_data" value="<?= base64_encode(json_encode($preview_errors)) ?>">

                        <?php if ($total_errors > 0): ?>
                            <label>
                                <input type="checkbox" name="skip_invalid" checked>
                                Skip invalid rows (import only valid rows)
                            </label>
                            <br><br>
                        <?php endif; ?>

                        <button type="submit">Confirm Import</button>
                        <a href="import.php"><button type="button">Cancel</button></a>
                    </form>

                    <?php endif; ?>

                </div>
            </div>
        </main>
    </div>
</body>
</html>
