<?php
session_start();

// Generate a fresh math captcha on every GET request
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $_SESSION['captcha_a']   = rand(2, 9);
    $_SESSION['captcha_b']   = rand(1, 9);
    $_SESSION['captcha_sum'] = $_SESSION['captcha_a'] + $_SESSION['captcha_b'];
}

$settings_file  = __DIR__ . '/data/org_settings.json';
$waitlist_file  = __DIR__ . '/data/waitlist.csv';
$inventory_file = __DIR__ . '/data/inventory.csv';

// Load org settings (used by both donate panel and the request form)
$org_defaults = [
    'org_name'             => 'Laptop Ministry',
    'drop_address_line1'   => '',
    'drop_address_line2'   => '',
    'drop_city_state_zip'  => '',
    'drop_hours'           => '',
    'drop_notes'           => '',
    'contact_name'         => '',
    'contact_email'        => '',
    'contact_phone'        => '',
    'tax_contact_name'     => '',
    'tax_contact_email'    => '',
    'tax_contact_phone'    => '',
    'tax_id'               => '',
    'welcome_intro'        => 'We collect older laptops, securely wipe them, refurbish them, and install an easy-to-use Linux operating system—then give them at no cost to Chicago neighbors who need one most.',
    'welcome_accept'       => "Laptops in working condition\nManufactured 2012 or newer",
    'welcome_dropoff_note' => 'Reception desk (1st floor) of the Inspiration Corporation building',
    'welcome_requests'     => "Messages about availability are usually answered on Sundays. Pickups (and special drop-off arrangements) can be coordinated once we confirm we have a suitable device for you. Because demand far outpaces our current capacity, we prioritize people with the greatest need.\n\nWe\xe2\x80\x99d love to see sister ministries like this one start in other neighborhoods and cities!\n\nThis effort became an official ministry of Peoples Church of Chicago in April 2023. For the latest details, to get in touch, or to learn how you can help, visit laptopministry.org",
    'welcome_tagline'      => "Thank you for helping reduce e-waste and close the digital divide\xe2\x80\x94one renewed laptop at a time!",
];
$org_raw = file_exists($settings_file) ? json_decode(file_get_contents($settings_file), true) : [];
$org = array_merge($org_defaults, is_array($org_raw) ? $org_raw : []);

// ---- Handle public laptop request form ----
// Processed before the login redirect so non-authenticated visitors can submit.
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['pub_action'] ?? '') === 'request_laptop') {
    // Honeypot: bots fill hidden "website" field; real users never see it
    if (!empty($_POST['website'])) {
        header('Location: index.php');
        exit;
    }

    // Math captcha check
    $captcha_input = intval(trim($_POST['captcha_input'] ?? ''));
    if ($captcha_input !== intval($_SESSION['captcha_sum'] ?? -1)) {
        header('Location: index.php?pub=captcha');
        exit;
    }

    $first   = trim($_POST['first_name'] ?? '');
    $last    = trim($_POST['last_name']  ?? '');
    $contact = trim($_POST['contact']    ?? '');
    $pref     = $_POST['laptop_pref'] ?? 'modern';
    $hw_raw   = $_POST['hw_needs']   ?? [];
    $hw_other = trim($_POST['hw_other']   ?? '');
    $comments = trim($_POST['comments']   ?? '');

    // Whitelist of valid hardware checkbox values
    $hw_labels = [
        'webcam'        => 'Webcam / camera',
        'bluetooth'     => 'Bluetooth',
        'wifi'          => 'Wi-Fi wireless',
        'ethernet'      => 'Ethernet port',
        'optical'       => 'Optical drive (CD/DVD)',
        'floppy'        => 'Floppy drive',
        'sd_card'       => 'SD card reader',
        'large_storage' => 'Large storage (256 GB+)',
        'hdmi'          => 'HDMI output',
        'vga'           => 'VGA output',
        'large_screen'  => 'Larger screen (15"+)',
        'numpad'        => 'Numeric keypad',
        'touchscreen'   => 'Touchscreen',
        'usb_c'         => 'USB-C port',
        'usb_a'         => 'Multiple USB-A ports',
        'good_battery'  => 'Good battery life',
        'lightweight'   => 'Lightweight / portable',
    ];
    $hw_needs = array_filter((array)$hw_raw, fn($v) => isset($hw_labels[$v]));

    if ($first !== '' && $last !== '' && $contact !== '') {
        $pref_labels = [
            'modern'  => 'Prefer: Modern laptop (2011 or newer)',
            'older'   => "Prefer: Older laptop (2005\xe2\x80\x932010)",
            'vintage' => 'Prefer: Vintage / hobby laptop (DOS / legacy use)',
        ];
        $pref_text = $pref_labels[$pref] ?? $pref_labels['modern'];

        $full_need = $pref_text;
        if (!empty($hw_needs) || $hw_other !== '') {
            $hw_items = array_map(fn($v) => $hw_labels[$v], $hw_needs);
            if ($hw_other !== '') $hw_items[] = 'Other: ' . $hw_other;
            $full_need .= "\nHardware preferences: " . implode(', ', $hw_items);
        }
        if ($comments !== '') {
            $full_need .= "\nComments: " . $comments;
        }

        // Read existing waitlist
        $wl_headers = ['number','first_name','last_name','contact','store_date','request_date',
                        'priority','computer_ready','inventory_id','computer_desc','computer_os',
                        'picked_up','pickup_date','need_desc','notes'];
        $wl_data = [];
        if (file_exists($waitlist_file)) {
            $fh = fopen($waitlist_file, 'r');
            fgetcsv($fh); // skip header row
            while (($row = fgetcsv($fh)) !== false) {
                if (count($row) >= 13) $wl_data[] = $row;
            }
            fclose($fh);
        }

        $max = 0;
        foreach ($wl_data as $row) {
            $n = intval($row[0] ?? 0);
            if ($n > $max) $max = $n;
        }

        $wl_data[] = [
            $max + 1,
            $first, $last, $contact,
            date('Ymd'),    // store_date
            date('Y-m-d'),  // request_date
            'N',            // priority
            'N',            // computer_ready
            '', '', '',     // inventory_id, computer_desc, computer_os
            'N', '',        // picked_up, pickup_date
            $full_need,
            'Public website signup — ' . date('Y-m-d'),
        ];

        $fh = fopen($waitlist_file, 'w');
        if (flock($fh, LOCK_EX)) {
            fputcsv($fh, $wl_headers);
            foreach ($wl_data as $row) fputcsv($fh, $row);
            flock($fh, LOCK_UN);
        }
        fclose($fh);

        header('Location: index.php?pub=ok');
        exit;
    } else {
        header('Location: index.php?pub=err');
        exit;
    }
}

$pub_msg = '';
if (isset($_GET['pub'])) {
    if ($_GET['pub'] === 'ok') {
        $pub_msg = 'success';
    } elseif ($_GET['pub'] === 'err') {
        $pub_msg = 'error';
    } elseif ($_GET['pub'] === 'captcha') {
        $pub_msg = 'captcha';
    }
}

// ---- Waitlist stats ----
$wl = [
    'total' => 0, 'waiting' => 0, 'provided' => 0, 'priority' => 0,
    'ready_not_picked' => 0, 'avg_rate' => 0,
    'first_pickup' => 'N/A', 'last_pickup' => 'N/A',
    'months' => []
];

if (file_exists($waitlist_file)) {
    $fh = fopen($waitlist_file, 'r');
    fgetcsv($fh);
    $pickup_dates = [];
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) < 15) $row = array_pad($row, 15, '');
        $wl['total']++;
        $picked    = trim($row[11]);
        $ready     = trim($row[7]);
        $prio      = trim($row[6]);
        $pdate_str = trim($row[12]);
        if ($picked === 'N') $wl['waiting']++;
        if ($prio === 'Y' && $picked === 'N') $wl['priority']++;
        if ($ready === 'Y' && $picked === 'N') $wl['ready_not_picked']++;
        if ($picked === 'Y' && !empty($pdate_str)) {
            $wl['provided']++;
            $date = false;
            foreach (['m/d/y', 'Y-m-d', 'Ymd'] as $fmt) {
                $d = DateTime::createFromFormat($fmt, $pdate_str);
                if ($d && $d->format($fmt) === $pdate_str) { $date = $d; break; }
            }
            if ($date) {
                $pickup_dates[] = $date;
                $mk = $date->format('Y-m');
                $wl['months'][$mk] = ($wl['months'][$mk] ?? 0) + 1;
            }
        }
    }
    fclose($fh);

    if (!empty($pickup_dates)) {
        $earliest = min($pickup_dates);
        $latest   = max($pickup_dates);
        $days = $latest->diff($earliest)->days;
        if ($days > 0) $wl['avg_rate'] = round($wl['provided'] / ($days / 30.44), 2);
        $wl['first_pickup'] = $earliest->format('m/d/Y');
        $wl['last_pickup']  = $latest->format('m/d/Y');
    }
}

ksort($wl['months']);
$recent_months  = array_slice($wl['months'], -6, 6, true);
$wl_fulfillment = $wl['total'] > 0 ? round(($wl['provided'] / $wl['total']) * 100) : 0;

// ---- Inventory stats ----
$inv = ['total' => 0, 'active' => 0, 'discarded' => 0,
        'total_value' => 0, 'active_value' => 0,
        'active_computers' => 0, 'assigned' => 0, 'unassigned' => 0,
        'vintage_computers' => 0];

$assigned_ids = [];
if (file_exists($waitlist_file)) {
    $fh = fopen($waitlist_file, 'r');
    fgetcsv($fh);
    while (($row = fgetcsv($fh)) !== false) {
        $iid = trim($row[8] ?? '');
        if (!empty($iid)) $assigned_ids[] = $iid;
    }
    fclose($fh);
}

if (file_exists($inventory_file)) {
    $fh = fopen($inventory_file, 'r');
    fgetcsv($fh);
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) < 10) continue;
        $inv['total']++;
        $val  = floatval($row[6] ?? 0);
        $disc = trim($row[8] ?? '');
        $inv['total_value'] += $val;
        if ($disc === 'Yes') {
            $inv['discarded']++;
        } else {
            $inv['active']++;
            $inv['active_value'] += $val;
            $item_type   = trim($row[10] ?? '');
            $item_status = trim($row[11] ?? '');
            if ($item_type !== 'Part' && $item_status === 'Vintage') {
                $inv['vintage_computers']++;
            }
            $is_ready    = $item_type !== 'Part'
                        && !in_array($item_status, ['Repair', 'Parts', 'Vintage']);
            if ($is_ready) {
                $inv['active_computers']++;
                if (in_array($row[0], $assigned_ids)) $inv['assigned']++;
                else $inv['unassigned']++;
            }
        }
    }
    fclose($fh);
}

$inv_assign_pct = $inv['active_computers'] > 0 ? round(($inv['assigned'] / $inv['active_computers']) * 100) : 0;
$inv_util_pct   = $inv['total']  > 0 ? round(($inv['active']   / $inv['total'])  * 100) : 0;

$ops_span = '';
if ($wl['first_pickup'] !== 'N/A' && $wl['last_pickup'] !== 'N/A') {
    $d1 = DateTime::createFromFormat('m/d/Y', $wl['first_pickup']);
    $d2 = DateTime::createFromFormat('m/d/Y', $wl['last_pickup']);
    if ($d1 && $d2) {
        $diff     = $d2->diff($d1);
        $ops_span = $diff->y . ' yr' . ($diff->y !== 1 ? 's' : '') . ', ' . $diff->m . ' mo';
    }
}

// Check if the donate panel has enough info to display
$has_donate_info = trim($org['drop_address_line1']) !== '' || trim($org['contact_email']) !== '' || trim($org['contact_phone']) !== '';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Laptop Ministry</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
    <style>
        /* ---- Public action panels ---- */
        .pub-panels {
            display: flex;
            gap: 14px;
            flex-wrap: wrap;
            margin-bottom: 16px;
        }
        .pub-panel {
            flex: 1;
            min-width: 280px;
        }
        .pub-panel .window-title {
            font-size: 14px;
        }
        /* Radio button rows */
        .radio-group { margin: 8px 0; }
        .radio-group label {
            display: flex;
            align-items: flex-start;
            gap: 8px;
            font-weight: normal;
            cursor: pointer;
            padding: 5px 6px;
            border: 1px solid transparent;
            margin-bottom: 3px;
        }
        .radio-group label:hover { background: #d8e8f8; border-color: #a0c0e0; }
        .radio-group input[type="radio"] {
            width: auto;
            margin: 2px 0 0;
            flex-shrink: 0;
            border: none;
        }
        .radio-label-main { font-weight: bold; font-size: 13px; display: block; }
        .radio-label-sub  { font-size: 11px; color: #555; display: block; }
        /* Donate panel details */
        .donate-detail { margin: 10px 0; font-size: 13px; line-height: 1.6; }
        .donate-detail strong { display: block; font-size: 11px; text-transform: uppercase; letter-spacing: 0.5px; color: #444; margin-bottom: 2px; }
        .donate-section { border-top: 1px solid #b0b0b0; padding-top: 8px; margin-top: 8px; }
        .pub-success {
            background: #e8f5e9; border: 2px solid #006600; padding: 12px 14px;
            margin-bottom: 12px; font-size: 14px;
        }
        .pub-error {
            background: #ffebee; border: 2px solid #cc0000; padding: 12px 14px;
            margin-bottom: 12px; font-size: 14px;
        }
        /* Honeypot — must be invisible */
        .hp-field { display: none !important; }
        /* Compact form inputs on this page */
        .pub-panel input[type="text"],
        .pub-panel input[type="email"],
        .pub-panel input[type="tel"],
        .pub-panel textarea {
            font-size: 13px;
            padding: 4px 6px;
            margin-bottom: 6px;
        }
        .pub-panel textarea { min-height: 60px; resize: vertical; }
        .pub-panel .form-row { margin-bottom: 6px; }
        .pub-panel .form-row label { font-size: 12px; margin-bottom: 2px; }
        .pub-panel .form-inline { gap: 6px; }
        @media (max-width: 640px) {
            .pub-panel { min-width: 100%; }
        }
        /* Hardware checkboxes */
        .hw-group-title {
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            color: #555;
            margin: 10px 0 3px;
            border-bottom: 1px solid #ddd;
            padding-bottom: 2px;
        }
        .checkbox-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(152px, 1fr));
            gap: 2px 6px;
            margin-bottom: 4px;
        }
        .checkbox-grid label {
            display: flex;
            align-items: center;
            gap: 6px;
            font-size: 12px;
            font-weight: normal;
            padding: 3px 5px;
            cursor: pointer;
            border: 1px solid transparent;
        }
        .checkbox-grid label:hover { background: #d8e8f8; border-color: #a0c0e0; }
        .checkbox-grid input[type="checkbox"] {
            width: auto;
            margin: 0;
            flex-shrink: 0;
            border: none;
        }
        #hw-other-row { display: none; margin: 4px 0 6px; }
        #hw-other-row input { font-size: 12px; }
        /* Welcome modal */
        #intro-overlay {
            position: fixed;
            inset: 0;
            background: rgba(0,0,0,0.55);
            z-index: 9000;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 16px;
        }
        #intro-modal {
            max-width: 520px;
            width: 100%;
            max-height: 90vh;
            display: flex;
            flex-direction: column;
        }
        #intro-modal .window-content {
            overflow-y: auto;
            font-size: 13px;
            line-height: 1.6;
        }
        #intro-modal h4 {
            margin: 12px 0 4px;
            font-size: 13px;
        }
        #intro-modal ul {
            margin: 4px 0 4px 18px;
            padding: 0;
        }
        #intro-modal ul li { margin-bottom: 2px; }
        #intro-modal .drop-address {
            background: #f0f4f8;
            border: 1px solid #b0c4d8;
            padding: 7px 10px;
            margin: 6px 0;
            font-size: 12px;
            line-height: 1.7;
        }
        #intro-modal .intro-footer {
            margin-top: 12px;
            padding-top: 10px;
            border-top: 1px solid #ccc;
            font-size: 11px;
            color: #555;
            text-align: center;
        }
        #intro-modal .btn-row {
            margin-top: 12px;
            text-align: center;
        }
    </style>
</head>
<body>

<!-- ===================== WELCOME MODAL (new visitors) ===================== -->
<div id="intro-overlay" style="display:none;" role="dialog" aria-modal="true" aria-labelledby="intro-title">
    <div id="intro-modal" class="window">
        <div class="window-title" id="intro-title">&#128187; About Laptop Ministry</div>
        <div class="window-content">

            <?php if (trim($org['welcome_intro']) !== ''): ?>
            <p><?= nl2br(htmlspecialchars($org['welcome_intro'])) ?></p>
            <?php endif; ?>

            <?php
            $accept_items = array_filter(array_map('trim', explode("\n", $org['welcome_accept'])));
            if (!empty($accept_items)):
            ?>
            <h4>What we accept:</h4>
            <ul>
                <?php foreach ($accept_items as $item): ?>
                    <li><?= htmlspecialchars($item) ?></li>
                <?php endforeach; ?>
            </ul>
            <?php endif; ?>

            <?php
            $has_modal_address = trim($org['drop_address_line1']) !== '' || trim($org['drop_city_state_zip']) !== '';
            if ($has_modal_address || trim($org['welcome_dropoff_note']) !== ''):
            ?>
            <h4>Drop-off &mdash; 24/7:</h4>
            <div class="drop-address">
                <?php if (trim($org['welcome_dropoff_note']) !== ''): ?>
                    <?= htmlspecialchars($org['welcome_dropoff_note']) ?><br>
                <?php endif; ?>
                <?php if ($has_modal_address): ?>
                    <strong>
                    <?= htmlspecialchars($org['drop_address_line1']) ?>
                    <?php if (trim($org['drop_address_line2']) !== ''): ?>, <?= htmlspecialchars($org['drop_address_line2']) ?><?php endif; ?>
                    <?php if (trim($org['drop_city_state_zip']) !== ''): ?><br><?= htmlspecialchars($org['drop_city_state_zip']) ?><?php endif; ?>
                    </strong>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php if (trim($org['welcome_requests']) !== ''): ?>
            <h4>Want a refurbished laptop?</h4>
            <?php
            $req_paras = array_filter(explode("\n\n", $org['welcome_requests']));
            foreach ($req_paras as $para):
            ?>
                <p><?= nl2br(htmlspecialchars(trim($para))) ?></p>
            <?php endforeach; ?>
            <?php endif; ?>

            <?php if (trim($org['welcome_tagline']) !== ''): ?>
            <div class="intro-footer">
                <?= htmlspecialchars($org['welcome_tagline']) ?>
            </div>
            <?php endif; ?>

            <div class="btn-row">
                <button id="intro-ok" onclick="closeIntro()">OK</button>
                <span style="font-size:11px;color:#777;margin-left:10px;">or press <kbd>Esc</kbd></span>
            </div>
        </div>
    </div>
</div>

<div class="login-stats-wrapper">

    <!-- ===================== PUBLIC ACTION PANELS ===================== -->
    <?php if ($pub_msg === 'ok'): ?>
    <div class="pub-success">
        <strong>&#10003; Request received!</strong>
        Thank you &mdash; your name has been added to our waitlist. We will reach out when a laptop matching your preference becomes available.
    </div>
    <?php elseif ($pub_msg === 'error'): ?>
    <div class="pub-error">
        <strong>&#9888; Missing information.</strong>
        Please fill in your first name, last name, and a contact (phone or email) before submitting.
    </div>
    <?php elseif ($pub_msg === 'captcha'): ?>
    <div class="pub-error">
        <strong>&#9888; Incorrect answer.</strong>
        Please answer the math question correctly and try again &mdash; a new question has been generated.
    </div>
    <?php endif; ?>

    <div class="pub-panels">

        <!-- REQUEST A LAPTOP -->
        <div class="pub-panel">
            <div class="window">
                <div class="window-title">&#128187; Request a Laptop</div>
                <div class="window-content">
                    <p style="font-size:12px;margin:0 0 10px;">Fill out the form below and we will contact you when a laptop is available. All fields marked * are required.</p>
                    <form method="post" action="index.php">
                        <input type="hidden" name="pub_action" value="request_laptop">
                        <!-- Honeypot anti-spam -->
                        <div class="hp-field">
                            <label>Leave blank: <input type="text" name="website" tabindex="-1" autocomplete="off"></label>
                        </div>

                        <div class="form-inline">
                            <div class="form-row">
                                <label>First Name *</label>
                                <input type="text" name="first_name" required autocomplete="given-name" placeholder="First">
                            </div>
                            <div class="form-row">
                                <label>Last Name *</label>
                                <input type="text" name="last_name" required autocomplete="family-name" placeholder="Last">
                            </div>
                        </div>

                        <div class="form-row">
                            <label>Phone or Email *</label>
                            <input type="text" name="contact" required autocomplete="email" placeholder="Best way to reach you">
                        </div>

                        <div class="form-row">
                            <label style="font-weight:bold;font-size:12px;margin-bottom:4px;">Laptop Preference *</label>
                            <div class="radio-group">
                                <label>
                                    <input type="radio" name="laptop_pref" value="modern" checked>
                                    <span>
                                        <span class="radio-label-main">Modern &mdash; 2011 or newer</span>
                                        <span class="radio-label-sub">Current OS, everyday tasks, school, work, browsing</span>
                                    </span>
                                </label>
                                <label>
                                    <input type="radio" name="laptop_pref" value="older">
                                    <span>
                                        <span class="radio-label-main">Older &mdash; 2005 to 2010</span>
                                        <span class="radio-label-sub">Light use, Linux-friendly, budget setups</span>
                                    </span>
                                </label>
                                <label>
                                    <input type="radio" name="laptop_pref" value="vintage">
                                    <span>
                                        <span class="radio-label-main">Vintage / Hobby &mdash; pre-2005</span>
                                        <span class="radio-label-sub">DOS, legacy software, retro computing, collectors</span>
                                    </span>
                                </label>
                            </div>
                        </div>

                        <div class="form-row">
                            <label style="font-weight:bold;font-size:12px;margin-bottom:2px;">Hardware Preferences <em style="font-weight:normal;">(optional &mdash; check all that apply)</em></label>

                            <div class="hw-group-title">Connectivity &amp; Communication</div>
                            <div class="checkbox-grid">
                                <label><input type="checkbox" name="hw_needs[]" value="webcam"> Webcam / camera</label>
                                <label><input type="checkbox" name="hw_needs[]" value="bluetooth"> Bluetooth</label>
                                <label><input type="checkbox" name="hw_needs[]" value="wifi"> Wi-Fi wireless</label>
                                <label><input type="checkbox" name="hw_needs[]" value="ethernet"> Ethernet port</label>
                            </div>

                            <div class="hw-group-title">Drives &amp; Storage</div>
                            <div class="checkbox-grid">
                                <label><input type="checkbox" name="hw_needs[]" value="optical"> Optical drive (CD/DVD)</label>
                                <label><input type="checkbox" name="hw_needs[]" value="floppy"> Floppy drive <em style="font-size:10px;color:#666;">(vintage)</em></label>
                                <label><input type="checkbox" name="hw_needs[]" value="sd_card"> SD card reader</label>
                                <label><input type="checkbox" name="hw_needs[]" value="large_storage"> Large storage (256 GB+)</label>
                            </div>

                            <div class="hw-group-title">Display, Ports &amp; Input</div>
                            <div class="checkbox-grid">
                                <label><input type="checkbox" name="hw_needs[]" value="hdmi"> HDMI output</label>
                                <label><input type="checkbox" name="hw_needs[]" value="vga"> VGA output</label>
                                <label><input type="checkbox" name="hw_needs[]" value="large_screen"> Larger screen (15&Prime;+)</label>
                                <label><input type="checkbox" name="hw_needs[]" value="numpad"> Numeric keypad</label>
                                <label><input type="checkbox" name="hw_needs[]" value="touchscreen"> Touchscreen</label>
                                <label><input type="checkbox" name="hw_needs[]" value="usb_c"> USB-C port</label>
                                <label><input type="checkbox" name="hw_needs[]" value="usb_a"> Multiple USB-A ports</label>
                            </div>

                            <div class="hw-group-title">General</div>
                            <div class="checkbox-grid">
                                <label><input type="checkbox" name="hw_needs[]" value="good_battery" onchange="batteryWarn(this)"> Good battery life</label>
                                <label><input type="checkbox" name="hw_needs[]" value="lightweight"> Lightweight / portable</label>
                                <label><input type="checkbox" name="hw_needs[]" value="other" id="hw-other-cb" onchange="toggleOther(this)"> Other (specify below)</label>
                            </div>
                            <div id="battery-warn" style="display:none;margin:4px 0 2px;padding:6px 9px;background:#fff8dc;border:1px solid #cc8800;font-size:11px;color:#5a3a00;">
                                &#9888; <strong>Heads up:</strong> Donated laptops with working batteries are rare &mdash; selecting this will significantly extend your wait time. We recommend leaving this unchecked and planning to use the laptop plugged in or purchasing a replacement battery separately.
                            </div>
                            <div id="hw-other-row">
                                <input type="text" name="hw_other" placeholder="Describe other hardware needs...">
                            </div>
                        </div>

                        <div class="form-row">
                            <label>Comments <em style="font-weight:normal;">(optional)</em></label>
                            <textarea name="comments" placeholder="Any additional context &mdash; how you plan to use it, your situation, questions, etc."></textarea>
                        </div>

                        <div class="form-row">
                            <label>Spam check &mdash; what is <?= (int)($_SESSION['captcha_a'] ?? 3) ?> + <?= (int)($_SESSION['captcha_b'] ?? 4) ?>? *</label>
                            <input type="number" name="captcha_input" required min="0" max="99" placeholder="Enter the answer" style="max-width:100px;">
                        </div>

                        <button type="submit">Submit Request</button>
                    </form>
                </div>
            </div>
        </div>

        <!-- DONATE A LAPTOP -->
        <div class="pub-panel">
            <div class="window">
                <div class="window-title">&#127873; Donate a Laptop</div>
                <div class="window-content">
                    <p style="font-size:12px;margin:0 0 10px;">
                        We gratefully accept laptop and computer donations in any condition &mdash; working, for parts, or vintage.
                        Every donation helps someone in need.
                    </p>

                    <?php if ($has_donate_info): ?>

                        <?php if (trim($org['drop_address_line1']) !== ''): ?>
                        <div class="donate-detail">
                            <strong>Drop-off Address</strong>
                            <?= htmlspecialchars($org['drop_address_line1']) ?><br>
                            <?php if (trim($org['drop_address_line2']) !== ''): ?>
                                <?= htmlspecialchars($org['drop_address_line2']) ?><br>
                            <?php endif; ?>
                            <?= htmlspecialchars($org['drop_city_state_zip']) ?>
                        </div>
                        <?php endif; ?>

                        <?php if (trim($org['drop_hours']) !== ''): ?>
                        <div class="donate-detail">
                            <strong>Drop-off Hours</strong>
                            <?= nl2br(htmlspecialchars($org['drop_hours'])) ?>
                        </div>
                        <?php endif; ?>

                        <?php if (trim($org['drop_notes']) !== ''): ?>
                        <div class="donate-detail">
                            <?= nl2br(htmlspecialchars($org['drop_notes'])) ?>
                        </div>
                        <?php endif; ?>

                        <?php if (trim($org['contact_name']) !== '' || trim($org['contact_email']) !== '' || trim($org['contact_phone']) !== ''): ?>
                        <div class="donate-detail donate-section">
                            <strong>Contact for Donation Questions</strong>
                            <?php if (trim($org['contact_name'])  !== ''): ?><?= htmlspecialchars($org['contact_name'])  ?><br><?php endif; ?>
                            <?php if (trim($org['contact_email']) !== ''): ?><a href="mailto:<?= htmlspecialchars($org['contact_email']) ?>"><?= htmlspecialchars($org['contact_email']) ?></a><br><?php endif; ?>
                            <?php if (trim($org['contact_phone']) !== ''): ?><?= htmlspecialchars($org['contact_phone']) ?><br><?php endif; ?>
                        </div>
                        <?php endif; ?>

                        <?php if (trim($org['tax_contact_name']) !== '' || trim($org['tax_contact_email']) !== '' || trim($org['tax_id']) !== ''): ?>
                        <div class="donate-detail donate-section">
                            <strong>Tax-Deductible Donation / Receipt Requests</strong>
                            <?php if (trim($org['tax_id']) !== ''): ?>
                                <span style="font-size:12px;">EIN / Tax&nbsp;ID: <strong><?= htmlspecialchars($org['tax_id']) ?></strong></span><br>
                            <?php endif; ?>
                            <?php if (trim($org['tax_contact_name'])  !== ''): ?><?= htmlspecialchars($org['tax_contact_name'])  ?><br><?php endif; ?>
                            <?php if (trim($org['tax_contact_email']) !== ''): ?><a href="mailto:<?= htmlspecialchars($org['tax_contact_email']) ?>"><?= htmlspecialchars($org['tax_contact_email']) ?></a><br><?php endif; ?>
                            <?php if (trim($org['tax_contact_phone']) !== ''): ?><?= htmlspecialchars($org['tax_contact_phone']) ?><br><?php endif; ?>
                            <span style="font-size:11px;color:#555;">A tax-exempt donation letter can be requested after drop-off. Please include your name and contact when reaching out.</span>
                        </div>
                        <?php endif; ?>

                    <?php else: ?>
                        <p style="font-size:13px;color:#555;font-style:italic;">
                            Donation drop-off details are being updated &mdash; please check back soon or contact us for current information.
                        </p>
                    <?php endif; ?>
                </div>
            </div>
        </div>

    </div><!-- /.pub-panels -->

    <!-- ===================== PUBLIC STATS ===================== -->
    <div class="window">
        <div class="window-title">Waitlist Progress</div>
        <div class="window-content">
            <div class="stats-grid">
                <div class="stat-box">
                    <div class="stat-value"><?= $wl['waiting'] ?></div>
                    <div class="stat-label">People Waiting</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $wl['provided'] ?></div>
                    <div class="stat-label">Laptops Provided</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $wl['avg_rate'] ?></div>
                    <div class="stat-label">Laptops / Month</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $wl['priority'] ?></div>
                    <div class="stat-label">Priority Waiting</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $wl['ready_not_picked'] ?></div>
                    <div class="stat-label">Ready for Pickup</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $wl['total'] ?></div>
                    <div class="stat-label">Total Entries</div>
                </div>
            </div>

            <div class="progress-row">
                <div class="progress-label">
                    <span>Fulfillment Rate</span>
                    <span><?= $wl_fulfillment ?>% (<?= $wl['provided'] ?> of <?= $wl['total'] ?>)</span>
                </div>
                <div class="progress-bar-track">
                    <div class="progress-bar-fill bar-green" style="width:<?= $wl_fulfillment ?>%;"></div>
                </div>
            </div>

            <?php if ($wl['total'] > 0): ?>
            <div class="progress-row">
                <div class="progress-label">
                    <span>Waiting vs Served</span>
                    <span><?= $wl['waiting'] ?> waiting / <?= $wl['provided'] ?> served</span>
                </div>
                <div class="progress-bar-track">
                    <div class="progress-bar-fill bar-orange" style="width:<?= round(($wl['waiting'] / $wl['total']) * 100) ?>%;"></div>
                </div>
            </div>
            <?php endif; ?>

            <div style="display:flex;gap:15px;flex-wrap:wrap;margin-top:8px;">
                <div style="flex:1;min-width:150px;">
                    <strong style="font-size:12px;">Timeline</strong><br>
                    <span style="font-size:12px;">
                        First: <?= htmlspecialchars($wl['first_pickup']) ?><br>
                        Last: <?= htmlspecialchars($wl['last_pickup']) ?>
                        <?php if ($ops_span): ?><br>Span: <?= $ops_span ?><?php endif; ?>
                    </span>
                </div>
                <?php if (!empty($recent_months)): ?>
                <div style="flex:2;min-width:200px;">
                    <strong style="font-size:12px;">Recent Monthly Pickups</strong>
                    <table class="mini-table">
                        <tr>
                            <?php foreach ($recent_months as $m => $c): ?>
                                <th><?= date('M y', strtotime($m . '-01')) ?></th>
                            <?php endforeach; ?>
                        </tr>
                        <tr>
                            <?php foreach ($recent_months as $c): ?>
                                <td><strong><?= $c ?></strong></td>
                            <?php endforeach; ?>
                        </tr>
                    </table>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <div class="window" style="margin-top:10px;">
        <div class="window-title">Inventory Status</div>
        <div class="window-content">
            <div class="stats-grid">
                <div class="stat-box">
                    <div class="stat-value"><?= $inv['total'] ?></div>
                    <div class="stat-label">Total Items Received</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $inv['active_computers'] ?></div>
                    <div class="stat-label">Ready Computers</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $inv['vintage_computers'] ?></div>
                    <div class="stat-label">Vintage / Special</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $inv['assigned'] ?></div>
                    <div class="stat-label">Computers Assigned</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $inv['unassigned'] ?></div>
                    <div class="stat-label">Computers Available</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value">$<?= number_format($inv['total_value'], 0) ?></div>
                    <div class="stat-label">Total Value Received</div>
                </div>
                <div class="stat-box">
                    <div class="stat-value"><?= $inv['discarded'] ?></div>
                    <div class="stat-label">Discarded / Recycled</div>
                </div>
            </div>

            <div class="progress-row">
                <div class="progress-label">
                    <span>Computers / Laptops Assigned</span>
                    <span><?= $inv_assign_pct ?>% (<?= $inv['assigned'] ?> of <?= $inv['active_computers'] ?>)</span>
                </div>
                <div class="progress-bar-track">
                    <div class="progress-bar-fill bar-blue" style="width:<?= $inv_assign_pct ?>%;"></div>
                </div>
            </div>

            <div class="progress-row">
                <div class="progress-label">
                    <span>Utilization (Active vs Discarded)</span>
                    <span><?= $inv_util_pct ?>% active</span>
                </div>
                <div class="progress-bar-track">
                    <div class="progress-bar-fill bar-green" style="width:<?= $inv_util_pct ?>%;"></div>
                </div>
            </div>

            <div style="margin-top:8px;font-size:12px;">
                <strong>Active inventory value:</strong> $<?= number_format($inv['active_value'], 2) ?>
            </div>
        </div>
    </div>

    <div class="footer-credit">
        Laptop Ministry Accounting &mdash; <?= date('Y') ?> Peoples Church of Chicago
        &nbsp;&middot;&nbsp; <a href="login.php" style="color:inherit;opacity:0.7;">Staff Login</a>
    </div>

</div><!-- /.login-stats-wrapper -->
<script>
function toggleOther(cb) {
    document.getElementById('hw-other-row').style.display = cb.checked ? 'block' : 'none';
    if (!cb.checked) {
        document.querySelector('[name="hw_other"]').value = '';
    }
}
function batteryWarn(cb) {
    document.getElementById('battery-warn').style.display = cb.checked ? 'block' : 'none';
}
// Welcome modal — shown once per browser via localStorage
function closeIntro() {
    document.getElementById('intro-overlay').style.display = 'none';
    try { localStorage.setItem('lm_intro_seen', '1'); } catch(e) {}
}
(function() {
    var seen = false;
    try { seen = !!localStorage.getItem('lm_intro_seen'); } catch(e) {}
    if (!seen) {
        document.getElementById('intro-overlay').style.display = 'flex';
        document.getElementById('intro-ok').focus();
    }
})();
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        var overlay = document.getElementById('intro-overlay');
        if (overlay && overlay.style.display !== 'none') closeIntro();
    }
});
</script>
</body>
</html>
