<?php
require_once 'auth.php';

$csv_file = __DIR__ . '/data/inventory.csv';
$waitlist_file = __DIR__ . '/data/waitlist.csv';
$inv_headers = ['id','entry_date','date_received','donor_name','donor_contact','tax_letter','item_value','item_description','discarded','notes','item_type','item_status'];
$wl_headers = ['number','first_name','last_name','contact','store_date','request_date','priority','computer_ready','inventory_id','computer_desc','computer_os','picked_up','pickup_date','need_desc','notes'];

function generateUniqueID($existing_ids) {
    $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    do {
        $id = '';
        for ($i = 0; $i < 4; $i++) {
            $id .= $chars[random_int(0, strlen($chars) - 1)];
        }
    } while (in_array($id, $existing_ids));
    return $id;
}

function readInventory($file) {
    $data = [];
    if (!file_exists($file)) return $data;
    $fh = fopen($file, 'r');
    $headers = fgetcsv($fh);
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) >= 10) {
            $data[] = count($row) < 12 ? array_pad($row, 12, '') : $row;
        } elseif (count($row) > 0) {
            $data[] = array_pad($row, 12, '');
        }
    }
    fclose($fh);
    return $data;
}

function writeInventory($file, $data) {
    global $inv_headers;
    $fh = fopen($file, 'w');
    if (flock($fh, LOCK_EX)) {
        fputcsv($fh, $inv_headers);
        foreach ($data as $row) {
            fputcsv($fh, $row);
        }
        flock($fh, LOCK_UN);
    }
    fclose($fh);
}

$message = '';
$msg_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    require_editor();
    verify_csrf();
    $data = readInventory($csv_file);

    if ($_POST['action'] === 'add') {
        $existing_ids = array_column($data, 0);
        $new = [
            generateUniqueID($existing_ids),
            date('Y-m-d H:i:s'),
            trim($_POST['date_received'] ?? ''),
            trim($_POST['donor_name'] ?? ''),
            trim($_POST['donor_contact'] ?? ''),
            $_POST['tax_letter'] ?? 'No',
            trim($_POST['item_value'] ?? '0'),
            trim($_POST['item_description'] ?? ''),
            $_POST['discarded'] ?? 'No',
            trim($_POST['notes'] ?? ''),
            $_POST['item_type'] ?? 'Computer',
            $_POST['item_status'] ?? 'Ready'
        ];
        $data[] = $new;
        writeInventory($csv_file, $data);
        header('Location: inventory.php?msg=added');
        exit;
    }

    if ($_POST['action'] === 'edit' && isset($_POST['id'])) {
        $id = $_POST['id'];
        foreach ($data as &$row) {
            if ($row[0] === $id) {
                $row[2] = trim($_POST['date_received'] ?? '');
                $row[3] = trim($_POST['donor_name'] ?? '');
                $row[4] = trim($_POST['donor_contact'] ?? '');
                $row[5] = $_POST['tax_letter'] ?? 'No';
                $row[6] = trim($_POST['item_value'] ?? '0');
                $row[7] = trim($_POST['item_description'] ?? '');
                $row[8] = $_POST['discarded'] ?? 'No';
                $row[9] = trim($_POST['notes'] ?? '');
                $row[10] = $_POST['item_type'] ?? 'Computer';
                $row[11] = $_POST['item_status'] ?? 'Ready';
                break;
            }
        }
        unset($row);
        writeInventory($csv_file, $data);
        header('Location: inventory.php?msg=updated');
        exit;
    }

    if ($_POST['action'] === 'delete' && isset($_POST['delete_id'])) {
        $id = $_POST['delete_id'];
        $data = array_filter($data, function($row) use ($id) {
            return $row[0] !== $id;
        });
        writeInventory($csv_file, array_values($data));
        header('Location: inventory.php?msg=deleted');
        exit;
    }

    if ($_POST['action'] === 'assign' && isset($_POST['inv_id']) && isset($_POST['wl_number'])) {
        $inv_id = $_POST['inv_id'];
        $wl_num = $_POST['wl_number'];

        // Find inventory item description
        $inv_desc = '';
        foreach ($data as $row) {
            if ($row[0] === $inv_id) {
                $inv_desc = $row[7];
                break;
            }
        }

        // Update waitlist entry
        if (file_exists($waitlist_file)) {
            $wl_data = [];
            $wfh = fopen($waitlist_file, 'r');
            $wl_head = fgetcsv($wfh);
            while (($wr = fgetcsv($wfh)) !== false) {
                $wl_data[] = count($wr) >= 15 ? $wr : array_pad($wr, 15, '');
            }
            fclose($wfh);

            foreach ($wl_data as &$wr) {
                if ($wr[0] == $wl_num) {
                    $wr[7] = 'Y';           // computer_ready
                    $wr[8] = $inv_id;       // inventory_id
                    $wr[9] = $inv_desc;     // computer_desc
                    break;
                }
            }
            unset($wr);

            $wfh = fopen($waitlist_file, 'w');
            flock($wfh, LOCK_EX);
            fputcsv($wfh, $wl_headers);
            foreach ($wl_data as $wr) { fputcsv($wfh, $wr); }
            flock($wfh, LOCK_UN);
            fclose($wfh);
        }
        header('Location: inventory.php?msg=assigned');
        exit;
    }
}

if (isset($_GET['msg'])) {
    $msgs = ['added' => 'Item added.', 'updated' => 'Item updated.', 'deleted' => 'Item deleted.', 'assigned' => 'Item assigned to waitlist entry.'];
    $message = $msgs[$_GET['msg']] ?? '';
    $msg_type = 'success';
}

$data = readInventory($csv_file);

$edit_entry = null;
if (isset($_GET['edit'])) {
    foreach ($data as $row) {
        if ($row[0] === $_GET['edit']) {
            $edit_entry = $row;
            break;
        }
    }
}

// Load waitlist people who need a computer (not picked up, no inventory assigned)
$unassigned_people = [];
$assigned_inv_ids = [];
if (file_exists($waitlist_file)) {
    $wfh = fopen($waitlist_file, 'r');
    fgetcsv($wfh);
    while (($wr = fgetcsv($wfh)) !== false) {
        if (count($wr) < 15) $wr = array_pad($wr, 15, '');
        $inv_id = trim($wr[8]);
        if (!empty($inv_id)) $assigned_inv_ids[] = $inv_id;
        if ($wr[11] !== 'Y' && empty($inv_id)) {
            $unassigned_people[] = ['number' => $wr[0], 'name' => trim($wr[1] . ' ' . $wr[2]), 'need' => $wr[13]];
        }
    }
    fclose($wfh);
}

$token = csrf_token();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inventory - Laptop Ministry Accounting</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
</head>
<body>
    <div class="admin-wrapper">
        <?php include '_sidebar.php'; ?>
        <main class="content">
            <div class="window">
                <div class="window-title">Inventory Management</div>
                <div class="window-content">

                    <?php if ($message): ?>
                        <p class="<?= $msg_type ?>"><?= htmlspecialchars($message) ?></p>
                    <?php endif; ?>

                    <?php if ($edit_entry && can_edit()): ?>
                    <div class="modal-overlay" id="edit-modal">
                        <div class="modal-content">
                            <div class="window-title">Edit Item <?= htmlspecialchars($edit_entry[0]) ?></div>
                            <form method="post">
                                <input type="hidden" name="action" value="edit">
                                <input type="hidden" name="id" value="<?= htmlspecialchars($edit_entry[0]) ?>">
                                <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                <div class="form-row">
                                    <label>Date Received:</label>
                                    <input type="date" name="date_received" value="<?= htmlspecialchars($edit_entry[2]) ?>" required>
                                </div>
                                <div class="form-row">
                                    <label>Donor Name:</label>
                                    <input type="text" name="donor_name" value="<?= htmlspecialchars($edit_entry[3]) ?>" required>
                                </div>
                                <div class="form-row">
                                    <label>Donor Contact:</label>
                                    <input type="text" name="donor_contact" value="<?= htmlspecialchars($edit_entry[4]) ?>">
                                </div>
                                <div class="form-row">
                                    <label>Tax Letter:</label>
                                    <select name="tax_letter">
                                        <option value="Yes" <?= $edit_entry[5]==='Yes'?'selected':'' ?>>Yes</option>
                                        <option value="No" <?= $edit_entry[5]==='No'?'selected':'' ?>>No</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Item Value ($):</label>
                                    <input type="number" name="item_value" step="0.01" value="<?= htmlspecialchars($edit_entry[6]) ?>" required>
                                </div>
                                <div class="form-row">
                                    <label>Item Description:</label>
                                    <textarea name="item_description" required><?= htmlspecialchars($edit_entry[7]) ?></textarea>
                                </div>
                                <div class="form-row">
                                    <label>Item Type:</label>
                                    <select name="item_type">
                                        <option value="Computer" <?= ($edit_entry[10]??'')!=='Part'?'selected':'' ?>>Computer / Laptop</option>
                                        <option value="Part" <?= ($edit_entry[10]??'')==='Part'?'selected':'' ?>>Part / Accessory</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <?php $es = $edit_entry[11] ?? ''; ?>
                                    <label>Condition / Status:</label>
                                    <select name="item_status">
                                        <option value="Ready" <?= ($es===''||$es==='Ready')?'selected':'' ?>>Ready to Fulfill</option>
                                        <option value="Repair" <?= $es==='Repair'?'selected':'' ?>>Needs Repair</option>
                                        <option value="Parts" <?= $es==='Parts'?'selected':'' ?>>Used for Parts</option>
                                        <option value="Vintage" <?= $es==='Vintage'?'selected':'' ?>>Vintage / Special</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Discarded:</label>
                                    <select name="discarded">
                                        <option value="No" <?= $edit_entry[8]==='No'?'selected':'' ?>>No</option>
                                        <option value="Yes" <?= $edit_entry[8]==='Yes'?'selected':'' ?>>Yes</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Notes:</label>
                                    <textarea name="notes"><?= htmlspecialchars($edit_entry[9]) ?></textarea>
                                </div>
                                <button type="submit">Save Changes</button>
                                <a href="inventory.php"><button type="button">Cancel</button></a>
                            </form>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if (can_edit()): ?>
                    <fieldset>
                        <legend>Add New Item</legend>
                        <form method="post">
                            <input type="hidden" name="action" value="add">
                            <input type="hidden" name="csrf_token" value="<?= $token ?>">
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>Date Received:</label>
                                    <input type="date" name="date_received" required>
                                </div>
                                <div class="form-row">
                                    <label>Donor Name:</label>
                                    <input type="text" name="donor_name" required>
                                </div>
                                <div class="form-row">
                                    <label>Donor Contact:</label>
                                    <input type="text" name="donor_contact">
                                </div>
                            </div>
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>Tax Letter:</label>
                                    <select name="tax_letter">
                                        <option value="Yes">Yes</option>
                                        <option value="No" selected>No</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Item Value ($):</label>
                                    <input type="number" name="item_value" step="0.01" required>
                                </div>
                                <div class="form-row">
                                    <label>Item Type:</label>
                                    <select name="item_type">
                                        <option value="Computer" selected>Computer / Laptop</option>
                                        <option value="Part">Part / Accessory</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Condition / Status:</label>
                                    <select name="item_status">
                                        <option value="Ready" selected>Ready to Fulfill</option>
                                        <option value="Repair">Needs Repair</option>
                                        <option value="Parts">Used for Parts</option>
                                        <option value="Vintage">Vintage / Special</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Discarded:</label>
                                    <select name="discarded">
                                        <option value="No" selected>No</option>
                                        <option value="Yes">Yes</option>
                                    </select>
                                </div>
                            </div>
                            <div class="form-row">
                                <label>Item Description:</label>
                                <textarea name="item_description" required></textarea>
                            </div>
                            <div class="form-row">
                                <label>Notes:</label>
                                <textarea name="notes"></textarea>
                            </div>
                            <button type="submit">Add Item</button>
                        </form>
                    </fieldset>
                    <?php else: ?>
                    <p><em>Read-only access &mdash; auditor view.</em></p>
                    <?php endif; ?>

                    <h3>Inventory (<span id="inv-count-total"><?= count($data) ?></span> items)</h3>
                    <div class="table-search">
                        <label for="inv-search">Search:</label>
                        <input type="search" id="inv-search" placeholder="Filter by any field..." oninput="filterTable('inv-table', this.value, 'inv-shown')">
                        <span id="inv-shown" class="search-count"></span>
                        <a href="reconcile.php"><button type="button">&#9741; Reconcile Tool</button></a>
                    </div>
                    <div class="table-scroll">
                        <table class="data-table inv-table" id="inv-table">
                            <thead>
                                <tr>
                                    <th class="sortable" onclick="sortTable('inv-table',0)">ID</th>
                                    <th class="sortable" onclick="sortTable('inv-table',1)">Entry Date</th>
                                    <th class="sortable" onclick="sortTable('inv-table',2)">Received</th>
                                    <th class="sortable" onclick="sortTable('inv-table',3)">Donor</th>
                                    <th class="sortable" onclick="sortTable('inv-table',4)">Contact</th>
                                    <th class="sortable" onclick="sortTable('inv-table',5)">Tax</th>
                                    <th class="sortable" onclick="sortTable('inv-table',6)">Value</th>
                                    <th class="sortable" onclick="sortTable('inv-table',7)">Description</th>
                                    <th class="sortable" onclick="sortTable('inv-table',8)">Discarded</th>
                                    <th class="sortable" onclick="sortTable('inv-table',9)">Notes</th>
                                    <th class="sortable" onclick="sortTable('inv-table',10)">Type</th>
                                    <th class="sortable" onclick="sortTable('inv-table',11)">Status</th>
                                    <?php if (can_edit()): ?><th>Actions</th><?php endif; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($data as $row): ?>
                                <?php
                                    $is_part      = ($row[10] ?? '') === 'Part';
                                    $item_status  = $row[11] ?? '';
                                    $is_repair    = $item_status === 'Repair';
                                    $is_partsonly = $item_status === 'Parts';
                                    $is_vintage   = $item_status === 'Vintage';
                                    $is_fulfillable = !$is_part && !$is_repair && !$is_partsonly && $row[8] !== 'Yes';
                                    $row_class = $is_repair ? 'repair-row' : ($is_partsonly ? 'partsonly-row' : ($is_vintage ? 'vintage-row' : ($is_part ? 'part-row' : '')));
                                ?>
                                <tr class="<?= $row_class ?>">
                                    <?php for ($i = 0; $i < 10; $i++): ?>
                                        <td><?= htmlspecialchars($row[$i] ?? '') ?></td>
                                    <?php endfor; ?>
                                    <td>
                                        <?php if ($is_part): ?>
                                            <span class="type-badge type-badge-part">Part</span>
                                        <?php else: ?>
                                            <span class="type-badge type-badge-computer">Computer</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($is_repair): ?>
                                            <span class="type-badge status-badge-repair">Needs Repair</span>
                                        <?php elseif ($is_partsonly): ?>
                                            <span class="type-badge status-badge-parts">Parts Only</span>
                                        <?php elseif ($is_vintage): ?>
                                            <span class="type-badge status-badge-vintage">Vintage/Special</span>
                                        <?php else: ?>
                                            <span class="type-badge status-badge-ready">Ready</span>
                                        <?php endif; ?>
                                    </td>
                                    <?php if (can_edit()): ?>
                                    <td class="action-btns">
                                        <a href="?edit=<?= htmlspecialchars($row[0]) ?>"><button type="button">Edit</button></a>
                                        <?php if ($is_fulfillable && !in_array($row[0], $assigned_inv_ids) && !empty($unassigned_people)): ?>
                                            <button type="button" onclick="openAssign('<?= htmlspecialchars($row[0]) ?>', '<?= htmlspecialchars(addslashes($row[7])) ?>')">Assign</button>
                                        <?php elseif (in_array($row[0], $assigned_inv_ids)): ?>
                                            <span style="font-size:11px;color:#006600;">Assigned</span>
                                        <?php endif; ?>
                                        <form method="post" onsubmit="return confirm('Delete item <?= htmlspecialchars($row[0]) ?>?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="delete_id" value="<?= htmlspecialchars($row[0]) ?>">
                                            <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                            <button type="submit" class="btn-danger">Del</button>
                                        </form>
                                    </td>
                                    <?php endif; ?>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                </div>
            </div>
        </main>
    </div>
<!-- Assign Modal -->
<div class="modal-overlay" id="assign-modal" style="display:none;">
    <div class="modal-content">
        <div class="window-title" id="assign-title">Assign Item</div>
        <form method="post">
            <input type="hidden" name="action" value="assign">
            <input type="hidden" name="csrf_token" value="<?= $token ?>">
            <input type="hidden" name="inv_id" id="assign-inv-id">
            <div class="form-row">
                <label>Assign to:</label>
                <select name="wl_number" required>
                    <option value="">-- Select person --</option>
                    <?php foreach ($unassigned_people as $p): ?>
                        <option value="<?= htmlspecialchars($p['number']) ?>">#<?= htmlspecialchars($p['number']) ?> <?= htmlspecialchars($p['name']) ?><?= !empty($p['need']) ? ' (' . htmlspecialchars(substr($p['need'], 0, 40)) . ')' : '' ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit">Assign</button>
            <button type="button" onclick="closeAssign()">Cancel</button>
        </form>
    </div>
</div>

<script>
function openAssign(invId, invDesc) {
    document.getElementById('assign-inv-id').value = invId;
    document.getElementById('assign-title').textContent = 'Assign ' + invId + ' \u2014 ' + invDesc;
    document.getElementById('assign-modal').style.display = 'flex';
}
function closeAssign() {
    document.getElementById('assign-modal').style.display = 'none';
}

/* ---- Shared table search & sort utilities ---- */
var _sortState = {};

function filterTable(tableId, term, countId) {
    var tbody = document.querySelector('#' + tableId + ' tbody');
    if (!tbody) return;
    var rows = tbody.querySelectorAll('tr');
    var lower = term.toLowerCase().trim();
    var visible = 0;
    rows.forEach(function(row) {
        var show = !lower || row.textContent.toLowerCase().includes(lower);
        row.style.display = show ? '' : 'none';
        if (show) visible++;
    });
    if (countId) {
        var el = document.getElementById(countId);
        if (el) el.textContent = lower ? '(' + visible + ' shown)' : '';
    }
}

function sortTable(tableId, colIdx) {
    var table = document.getElementById(tableId);
    if (!table) return;
    var tbody = table.querySelector('tbody');
    var rows = Array.from(tbody.querySelectorAll('tr'));
    if (!_sortState[tableId]) _sortState[tableId] = {col: -1, dir: 1};
    var state = _sortState[tableId];
    state.dir = (state.col === colIdx) ? state.dir * -1 : 1;
    state.col = colIdx;

    rows.sort(function(a, b) {
        var aVal = a.cells[colIdx] ? a.cells[colIdx].textContent.trim() : '';
        var bVal = b.cells[colIdx] ? b.cells[colIdx].textContent.trim() : '';
        var aNum = parseFloat(aVal.replace(/[$,]/g, ''));
        var bNum = parseFloat(bVal.replace(/[$,]/g, ''));
        if (!isNaN(aNum) && !isNaN(bNum)) return (aNum - bNum) * state.dir;
        return aVal.localeCompare(bVal, undefined, {numeric: true}) * state.dir;
    });
    rows.forEach(function(r) { tbody.appendChild(r); });

    table.querySelectorAll('thead th').forEach(function(th, i) {
        th.classList.remove('sort-asc', 'sort-desc');
        if (i === colIdx) th.classList.add(state.dir === 1 ? 'sort-asc' : 'sort-desc');
    });
}
</script>
</body>
</html>
