<?php
session_start();

define('USERS_FILE',    __DIR__ . '/data/users.json');
define('ATTEMPTS_FILE', __DIR__ . '/data/login_attempts.json');
define('MAX_ATTEMPTS',  5);
define('LOCKOUT_SECS',  15 * 60);  // 15 minutes

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: login.php');
    exit;
}

$ip = $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';

// ---- Load attempt log ----
$log = [];
if (file_exists(ATTEMPTS_FILE)) {
    $log = json_decode(file_get_contents(ATTEMPTS_FILE), true) ?? [];
}

// Prune stale entries (unlocked entries older than 1 hour)
foreach ($log as $addr => $rec) {
    if (empty($rec['locked_until']) && (time() - ($rec['last_attempt'] ?? 0)) > 3600) {
        unset($log[$addr]);
    }
}

$rec = $log[$ip] ?? ['attempts' => 0, 'locked_until' => 0, 'last_attempt' => 0];

// ---- Check lockout ----
if (!empty($rec['locked_until']) && time() < $rec['locked_until']) {
    $mins = ceil(($rec['locked_until'] - time()) / 60);
    header('Location: login.php?error=locked&mins=' . $mins);
    exit;
} elseif (!empty($rec['locked_until']) && time() >= $rec['locked_until']) {
    // Lockout expired — reset
    $rec = ['attempts' => 0, 'locked_until' => 0, 'last_attempt' => 0];
}

// ---- Math captcha check ----
$captcha_input = intval(trim($_POST['captcha_input'] ?? ''));
if ($captcha_input !== intval($_SESSION['login_captcha_sum'] ?? -1)) {
    header('Location: login.php?error=captcha');
    exit;
}

// ---- Credential check ----
$username = trim($_POST['username'] ?? '');
$password = $_POST['password'] ?? '';

if (!file_exists(USERS_FILE)) {
    header('Location: login.php?error=1');
    exit;
}

$users     = json_decode(file_get_contents(USERS_FILE), true) ?? [];
$logged_in = false;

foreach ($users as $user) {
    if ($user['username'] === $username && password_verify($password, $user['hash'])) {
        session_regenerate_id(true);
        $_SESSION['user_id']  = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['role']     = $user['role'];
        $logged_in = true;
        break;
    }
}

if ($logged_in) {
    // Clear attempt record on success
    unset($log[$ip]);
    file_put_contents(ATTEMPTS_FILE, json_encode($log, JSON_PRETTY_PRINT), LOCK_EX);
    header('Location: dashboard.php');
    exit;
}

// ---- Failed login — record attempt ----
$rec['attempts']++;
$rec['last_attempt'] = time();

if ($rec['attempts'] >= MAX_ATTEMPTS) {
    $rec['locked_until'] = time() + LOCKOUT_SECS;
    $rec['attempts']     = 0;
    $log[$ip] = $rec;
    file_put_contents(ATTEMPTS_FILE, json_encode($log, JSON_PRETTY_PRINT), LOCK_EX);
    header('Location: login.php?error=locked&mins=15');
    exit;
}

$log[$ip] = $rec;
file_put_contents(ATTEMPTS_FILE, json_encode($log, JSON_PRETTY_PRINT), LOCK_EX);

$remaining = MAX_ATTEMPTS - $rec['attempts'];
header('Location: login.php?error=1&left=' . $remaining);
exit;
