<?php
require_once 'auth.php';
require_editor();

$csv_inv = __DIR__ . '/data/inventory.csv';
$csv_wl  = __DIR__ . '/data/waitlist.csv';

$inv_headers = ['id','entry_date','date_received','donor_name','donor_contact','tax_letter','item_value','item_description','discarded','notes'];
$wl_headers  = ['number','first_name','last_name','contact','store_date','request_date','priority','computer_ready','inventory_id','computer_desc','computer_os','picked_up','pickup_date','need_desc','notes'];

function rec_readCSV($file, $minCols) {
    $data = [];
    if (!file_exists($file)) return $data;
    $fh = fopen($file, 'r');
    fgetcsv($fh);
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) > 0) {
            $data[] = count($row) >= $minCols ? $row : array_pad($row, $minCols, '');
        }
    }
    fclose($fh);
    return $data;
}

function rec_writeWL($file, $data, $headers) {
    $fh = fopen($file, 'w');
    if (flock($fh, LOCK_EX)) {
        fputcsv($fh, $headers);
        foreach ($data as $row) fputcsv($fh, $row);
        flock($fh, LOCK_UN);
    }
    fclose($fh);
}

$message  = '';
$msg_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    verify_csrf();
    $wl_data  = rec_readCSV($csv_wl, 15);
    $inv_data = rec_readCSV($csv_inv, 10);

    // Build inv ID → description map
    $inv_map = [];
    foreach ($inv_data as $row) $inv_map[$row[0]] = $row[7];

    if ($_POST['action'] === 'relink') {
        // Update a waitlist entry's inventory_id to a real inventory ID
        $wl_num    = (string)($_POST['wl_number'] ?? '');
        $new_inv   = trim($_POST['new_inv_id'] ?? '');
        foreach ($wl_data as &$row) {
            if ((string)$row[0] === $wl_num) {
                $row[8] = $new_inv;
                if ($new_inv && isset($inv_map[$new_inv])) {
                    $row[7] = 'Y';          // computer_ready
                    $row[9] = $inv_map[$new_inv]; // computer_desc
                }
                break;
            }
        }
        unset($row);
        rec_writeWL($csv_wl, $wl_data, $wl_headers);
        header('Location: reconcile.php?msg=relinked');
        exit;
    }

    if ($_POST['action'] === 'clear_inv') {
        // Remove invalid inventory_id from a single waitlist entry
        $wl_num = (string)($_POST['wl_number'] ?? '');
        foreach ($wl_data as &$row) {
            if ((string)$row[0] === $wl_num) {
                $row[8] = '';
                break;
            }
        }
        unset($row);
        rec_writeWL($csv_wl, $wl_data, $wl_headers);
        header('Location: reconcile.php?msg=cleared');
        exit;
    }

    if ($_POST['action'] === 'bulk_clear_pickedup') {
        // Clear invalid inventory_id from ALL picked-up waitlist entries with orphaned refs
        $inv_ids = array_column($inv_data, 0);
        $cleared = 0;
        foreach ($wl_data as &$row) {
            $inv_id = trim($row[8]);
            if (!empty($inv_id) && !in_array($inv_id, $inv_ids) && $row[11] === 'Y') {
                $row[8] = '';
                $cleared++;
            }
        }
        unset($row);
        rec_writeWL($csv_wl, $wl_data, $wl_headers);
        header('Location: reconcile.php?msg=bulk_cleared&n=' . $cleared);
        exit;
    }

    if ($_POST['action'] === 'link_to_wl') {
        // Link an unmatched inventory item to a waitlist entry
        $inv_id = trim($_POST['inv_id'] ?? '');
        $wl_num = (string)($_POST['wl_number'] ?? '');
        $inv_desc = $inv_map[$inv_id] ?? '';
        foreach ($wl_data as &$row) {
            if ((string)$row[0] === $wl_num) {
                $row[7] = 'Y';
                $row[8] = $inv_id;
                $row[9] = $inv_desc;
                break;
            }
        }
        unset($row);
        rec_writeWL($csv_wl, $wl_data, $wl_headers);
        header('Location: reconcile.php?msg=linked');
        exit;
    }
}

// Flash messages
if (isset($_GET['msg'])) {
    $msgs = [
        'relinked'     => 'Waitlist entry relinked to inventory item.',
        'cleared'      => 'Invalid inventory reference cleared.',
        'linked'       => 'Inventory item linked to waitlist entry.',
        'bulk_cleared' => 'Bulk clear: ' . intval($_GET['n'] ?? 0) . ' orphaned reference(s) removed from picked-up entries.',
    ];
    $message  = $msgs[$_GET['msg']] ?? '';
    $msg_type = 'success';
}

// ------ Build reconciliation data ------
$inv_data = rec_readCSV($csv_inv, 10);
$wl_data  = rec_readCSV($csv_wl, 15);

// Index inventory by ID
$inv_id_set = [];
foreach ($inv_data as $row) $inv_id_set[$row[0]] = $row;

// Scan waitlist: separate valid refs from orphaned refs
$valid_wl_inv_ids = []; // inventory IDs that are validly referenced
$orphaned_wl      = []; // waitlist rows whose inventory_id doesn't exist in inventory

foreach ($wl_data as $row) {
    $inv_id = trim($row[8]);
    if ($inv_id === '') continue;
    if (isset($inv_id_set[$inv_id])) {
        $valid_wl_inv_ids[] = $inv_id;
    } else {
        $orphaned_wl[] = $row;
    }
}

// Inventory items not referenced by any valid waitlist link (not discarded)
$unmatched_inv      = []; // fulfillable computers that need a link
$unmatched_parts    = []; // item_type=Part accessories — no link needed
$unmatched_nonfulf  = []; // computers marked Repair or Parts Only — no link needed
foreach ($inv_data as $row) {
    if ($row[8] === 'Yes') continue; // discarded
    if (in_array($row[0], $valid_wl_inv_ids)) continue; // already linked
    $item_type   = $row[10] ?? '';
    $item_status = $row[11] ?? '';
    if ($item_type === 'Part') {
        $unmatched_parts[] = $row;
    } elseif ($item_status === 'Repair' || $item_status === 'Parts') {
        $unmatched_nonfulf[] = $row;
    } else {
        $unmatched_inv[] = $row;
    }
}

// Waitlist people with no inventory assigned, not picked up (candidates for linking)
$open_wl = [];
foreach ($wl_data as $row) {
    if (trim($row[8]) === '' && $row[11] !== 'Y') {
        $open_wl[] = ['number' => $row[0], 'name' => trim($row[1] . ' ' . $row[2]), 'need' => substr($row[13], 0, 50)];
    }
}

// Orphaned: split into picked-up vs still-pending
$orphaned_pickedup = array_filter($orphaned_wl, fn($r) => $r[11] === 'Y');
$orphaned_pending  = array_filter($orphaned_wl, fn($r) => $r[11] !== 'Y');

$token = csrf_token();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reconcile - Laptop Ministry Accounting</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
</head>
<body>
<div class="admin-wrapper">
    <?php include '_sidebar.php'; ?>
    <main class="content">
        <div class="window">
            <div class="window-title">Inventory &harr; Waitlist Reconciliation</div>
            <div class="window-content">

                <?php if ($message): ?>
                    <p class="<?= $msg_type ?>"><?= htmlspecialchars($message) ?></p>
                <?php endif; ?>

                <p style="font-size:13px;margin-bottom:10px;">
                    This tool finds mismatches between the inventory and waitlist &mdash; specifically waitlist entries
                    that reference inventory IDs that no longer exist (e.g. old numeric IDs from before the current
                    system), and inventory items that have no matching waitlist link.
                </p>

                <!-- Summary -->
                <div class="reconcile-summary">
                    <div class="reconcile-stat">
                        <div class="rs-val <?= count($orphaned_wl) > 0 ? 'warn' : 'ok' ?>"><?= count($orphaned_wl) ?></div>
                        <div class="rs-label">Orphaned WL References</div>
                    </div>
                    <div class="reconcile-stat">
                        <div class="rs-val <?= count($orphaned_pickedup) > 0 ? 'warn' : 'ok' ?>"><?= count($orphaned_pickedup) ?></div>
                        <div class="rs-label">Orphaned (Picked Up)</div>
                    </div>
                    <div class="reconcile-stat">
                        <div class="rs-val <?= count($orphaned_pending) > 0 ? 'warn' : 'ok' ?>"><?= count($orphaned_pending) ?></div>
                        <div class="rs-label">Orphaned (Still Open)</div>
                    </div>
                    <div class="reconcile-stat">
                        <div class="rs-val"><?= count($unmatched_inv) ?></div>
                        <div class="rs-label">Unmatched Computers</div>
                    </div>
                    <div class="reconcile-stat">
                        <div class="rs-val"><?= count($unmatched_nonfulf) ?></div>
                        <div class="rs-label">Non-Fulfillable (OK)</div>
                    </div>
                    <div class="reconcile-stat">
                        <div class="rs-val"><?= count($unmatched_parts) ?></div>
                        <div class="rs-label">Unlinked Parts (OK)</div>
                    </div>
                    <div class="reconcile-stat">
                        <div class="rs-val ok"><?= count($valid_wl_inv_ids) ?></div>
                        <div class="rs-label">Valid WL&harr;Inv Links</div>
                    </div>
                </div>

                <!-- ====== Section 1: Orphaned picked-up entries ====== -->
                <div class="reconcile-section">
                    <h4>1. Orphaned References &mdash; Already Picked Up (<?= count($orphaned_pickedup) ?>)</h4>
                    <p style="font-size:12px;margin:0 0 6px;">
                        These waitlist entries are marked <strong>Picked Up = Y</strong> but reference inventory IDs
                        that don&rsquo;t exist (old system). You can either
                        <em>relink</em> them to the correct inventory item, or <em>clear</em> the bad reference.
                    </p>

                    <?php if (!empty($orphaned_pickedup)): ?>
                    <form method="post" onsubmit="return confirm('Clear all <?= count($orphaned_pickedup) ?> orphaned reference(s) from picked-up entries?');">
                        <input type="hidden" name="action" value="bulk_clear_pickedup">
                        <input type="hidden" name="csrf_token" value="<?= $token ?>">
                        <button type="submit" style="margin-bottom:8px;">&#10005; Bulk Clear All <?= count($orphaned_pickedup) ?> Orphaned Picked-Up References</button>
                    </form>
                    <?php endif; ?>

                    <?php foreach ($orphaned_pickedup as $row): ?>
                    <div class="recon-row">
                        <div class="recon-info">
                            <strong>#<?= htmlspecialchars($row[0]) ?> &mdash; <?= htmlspecialchars(trim($row[1] . ' ' . $row[2])) ?></strong>
                            <span class="badge badge-done">Picked Up</span><br>
                            Bad Inv ID: <code><?= htmlspecialchars($row[8]) ?></code>
                            &nbsp;|&nbsp; Desc on file: <?= htmlspecialchars($row[9]) ?: '<em>none</em>' ?><br>
                            Need: <?= htmlspecialchars(substr($row[13], 0, 80)) ?>
                        </div>
                        <div class="recon-actions">
                            <!-- Relink -->
                            <form method="post" style="display:inline-flex;gap:4px;align-items:center;">
                                <input type="hidden" name="action" value="relink">
                                <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                <input type="hidden" name="wl_number" value="<?= htmlspecialchars($row[0]) ?>">
                                <select name="new_inv_id" required>
                                    <option value="">-- Select inventory item --</option>
                                    <?php foreach ($inv_data as $inv): ?>
                                        <option value="<?= htmlspecialchars($inv[0]) ?>">
                                            <?= htmlspecialchars($inv[0]) ?> &mdash; <?= htmlspecialchars(substr($inv[7], 0, 50)) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <button type="submit">Relink</button>
                            </form>
                            &nbsp;
                            <!-- Clear -->
                            <form method="post" style="display:inline;" onsubmit="return confirm('Clear bad inventory ref from #<?= htmlspecialchars($row[0]) ?>?');">
                                <input type="hidden" name="action" value="clear_inv">
                                <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                <input type="hidden" name="wl_number" value="<?= htmlspecialchars($row[0]) ?>">
                                <button type="submit" class="btn-danger">Clear</button>
                            </form>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    <?php if (empty($orphaned_pickedup)): ?>
                        <p style="font-size:12px;color:#006600;">&#10003; No orphaned picked-up references.</p>
                    <?php endif; ?>
                </div>

                <!-- ====== Section 2: Orphaned open/pending entries ====== -->
                <div class="reconcile-section">
                    <h4>2. Orphaned References &mdash; Still Open / Not Picked Up (<?= count($orphaned_pending) ?>)</h4>
                    <p style="font-size:12px;margin:0 0 6px;">
                        These waitlist entries have an inventory_id set but it doesn&rsquo;t match any current
                        inventory item, and they haven&rsquo;t been picked up yet. Relink to the correct item
                        or clear the reference.
                    </p>

                    <?php foreach ($orphaned_pending as $row): ?>
                    <div class="recon-row">
                        <div class="recon-info">
                            <strong>#<?= htmlspecialchars($row[0]) ?> &mdash; <?= htmlspecialchars(trim($row[1] . ' ' . $row[2])) ?></strong>
                            <span class="badge badge-pending">Not Picked Up</span><br>
                            Bad Inv ID: <code><?= htmlspecialchars($row[8]) ?></code>
                            &nbsp;|&nbsp; Desc on file: <?= htmlspecialchars($row[9]) ?: '<em>none</em>' ?><br>
                            Need: <?= htmlspecialchars(substr($row[13], 0, 80)) ?>
                        </div>
                        <div class="recon-actions">
                            <form method="post" style="display:inline-flex;gap:4px;align-items:center;">
                                <input type="hidden" name="action" value="relink">
                                <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                <input type="hidden" name="wl_number" value="<?= htmlspecialchars($row[0]) ?>">
                                <select name="new_inv_id" required>
                                    <option value="">-- Select inventory item --</option>
                                    <?php foreach ($unmatched_inv as $inv): ?>
                                        <option value="<?= htmlspecialchars($inv[0]) ?>">
                                            <?= htmlspecialchars($inv[0]) ?> &mdash; <?= htmlspecialchars(substr($inv[7], 0, 50)) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <button type="submit">Relink</button>
                            </form>
                            &nbsp;
                            <form method="post" style="display:inline;" onsubmit="return confirm('Clear bad inventory ref from #<?= htmlspecialchars($row[0]) ?>?');">
                                <input type="hidden" name="action" value="clear_inv">
                                <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                <input type="hidden" name="wl_number" value="<?= htmlspecialchars($row[0]) ?>">
                                <button type="submit" class="btn-danger">Clear</button>
                            </form>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    <?php if (empty($orphaned_pending)): ?>
                        <p style="font-size:12px;color:#006600;">&#10003; No orphaned open references.</p>
                    <?php endif; ?>
                </div>

                <!-- ====== Section 3: Unmatched inventory items ====== -->
                <div class="reconcile-section">
                    <h4>3. Unmatched Computers &mdash; No Waitlist Link (<?= count($unmatched_inv) ?>)</h4>
                    <p style="font-size:12px;margin:0 0 6px;">
                        These <strong>Computer/Laptop</strong> inventory items are not referenced by any waitlist entry and are not discarded.
                        If a waitlist person received one, link it here. Items shown as &ldquo;Assigned&rdquo;
                        on the inventory page will update automatically once linked.
                        Parts/accessories are excluded from this section as they don&rsquo;t count toward fulfillment.
                    </p>

                    <div class="table-search">
                        <label for="inv3-search">Filter:</label>
                        <input type="search" id="inv3-search" placeholder="Search unmatched items..." oninput="filterReconItems(this.value)">
                    </div>

                    <div id="unmatched-inv-list">
                    <?php foreach ($unmatched_inv as $inv): ?>
                    <div class="recon-row" data-text="<?= htmlspecialchars(strtolower($inv[0] . ' ' . $inv[3] . ' ' . $inv[7])) ?>">
                        <div class="recon-info">
                            <strong><?= htmlspecialchars($inv[0]) ?></strong>
                            &nbsp;&mdash;&nbsp;<?= htmlspecialchars($inv[7]) ?><br>
                            Donor: <?= htmlspecialchars($inv[3]) ?>
                            &nbsp;|&nbsp; Received: <?= htmlspecialchars($inv[2]) ?>
                            &nbsp;|&nbsp; Value: $<?= htmlspecialchars($inv[6]) ?>
                        </div>
                        <?php if (!empty($open_wl)): ?>
                        <div class="recon-actions">
                            <form method="post" style="display:inline-flex;gap:4px;align-items:center;">
                                <input type="hidden" name="action" value="link_to_wl">
                                <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                <input type="hidden" name="inv_id" value="<?= htmlspecialchars($inv[0]) ?>">
                                <select name="wl_number" required>
                                    <option value="">-- Assign to person --</option>
                                    <?php foreach ($open_wl as $p): ?>
                                        <option value="<?= htmlspecialchars($p['number']) ?>">
                                            #<?= htmlspecialchars($p['number']) ?> <?= htmlspecialchars($p['name']) ?>
                                            <?= !empty($p['need']) ? ' (' . htmlspecialchars($p['need']) . ')' : '' ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <button type="submit">Link</button>
                            </form>
                        </div>
                        <?php else: ?>
                        <div class="recon-actions" style="color:#666;font-size:12px;">No open waitlist entries to link</div>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                    </div>

                    <?php if (empty($unmatched_inv)): ?>
                        <p style="font-size:12px;color:#006600;">&#10003; All non-discarded inventory items are matched to waitlist entries.</p>
                    <?php endif; ?>
                </div>

                <p style="margin-top:16px;">
                    <a href="inventory.php"><button type="button">&larr; Inventory</button></a>
                    &nbsp;
                    <a href="waitlist.php"><button type="button">&larr; Waitlist</button></a>
                </p>

            </div>
        </div>
    </main>
</div>
<script>
function filterReconItems(term) {
    var lower = term.toLowerCase().trim();
    document.querySelectorAll('#unmatched-inv-list .recon-row').forEach(function(row) {
        var text = row.dataset.text || row.textContent.toLowerCase();
        row.style.display = (!lower || text.includes(lower)) ? '' : 'none';
    });
}
</script>
</body>
</html>
