<?php
require_once 'auth.php';

$waitlist_file  = __DIR__ . '/data/waitlist.csv';
$inventory_file = __DIR__ . '/data/inventory.csv';

// Demographic keyword categories — order matters (first match wins; catch-all last).
// Vintage / Special Interest is also detected via inventory status (see $vintage_inv_ids).
$demo_categories = [
    'Education / School / College' => [
        'school', 'college', 'university', 'community college', 'high school', 'middle school',
        'elementary', 'preschool', 'k-12', 'homeschool', 'trade school', 'vocational',
        'education', 'student', 'homework', 'coursework', 'study', 'studying', 'academic',
        'learning', 'e-learning', 'online course', 'distance learning', 'degree', 'diploma',
        'certificate', 'certification', 'scholarship', 'tuition', 'class', 'training',
        'child', 'kid', 'youth', 'teen', 'teenager', 'young adult',
    ],
    'Homeless / Housing'           => [
        'homeless', 'unhoused', 'no fixed address', 'no permanent',
        'shelter', 'emergency shelter', 'crisis housing', 'crisis center',
        'housing', 'transitional housing', 'transitional living', 'halfway house',
        'group home', 'temporary housing', 'between homes', 'evict', 'eviction',
        'couch surfing', 'couch surf', 'displaced', 'motel', 'living in car',
        'living in vehicle', 'tent', 'domestic violence', 'dv shelter',
    ],
    'Job / Employment'             => [
        'job', 'work', 'employment', 'employ', 'unemployed', 'unemploy', 'jobless',
        'career', 'resume', 'cv', 'interview', 'job search', 'job hunting',
        'job application', 'job training', 'workforce', 'looking for work',
        'hire', 'hired', 'laid off', 'furlough', 'layoff',
        'remote work', 'work from home', 'wfh', 'side hustle', 'gig',
        'freelance', 'contractor', 'self-employed', 'entrepreneur', 'startup',
        'income', 'paycheck', 'wages', 'salary', 'temp work', 'seasonal work',
    ],
    'Recovery / Mental Health'     => [
        'recovery', 'rehab', 'rehabilitation', 'detox', 'sober living', 'sober',
        'sobriety', 'clean and sober', 'drug free', 'aa', 'na', '12-step', 'twelve step',
        'sponsor', 'relapse', 'outpatient', 'inpatient',
        'substance', 'substance abuse', 'addiction', 'drug', 'alcohol', 'alcoholism',
        'mental health', 'mental illness', 'therapy', 'therapist', 'counseling',
        'counselor', 'psychiatric', 'psychiatry', 'depression', 'anxiety',
        'ptsd', 'trauma', 'group therapy', 'wellness', 'treatment',
    ],
    'Gaming / Entertainment'       => [
        'gaming', 'gamer', 'game', 'video game', 'esports', 'e-sports',
        'minecraft', 'fortnite', 'roblox', 'steam',
        'twitch', 'youtube', 'streaming', 'stream',
        'fun', 'entertainment', 'entertain', 'hobby', 'leisure', 'recreation',
        'music', 'movies', 'films', 'creative', 'art', 'drawing', 'photography',
        'personal use', 'personal computer', 'just want', 'just need',
    ],
    'Vintage / Special Interest'   => [
        'vintage', 'retro', 'classic', 'collector', 'collecting', 'collection',
        'antique', 'old school', 'oldschool', 'throwback', 'historic', 'historical',
        'nostalgia', 'nostalgic', 'rare', 'heirloom', 'special interest',
        'classic computer', 'old computer', 'legacy',
    ],
    'Unspecified / Just Need'      => [],
];

function parseDate($str) {
    $str = trim($str);
    if ($str === '') return null;
    foreach (['Y-m-d', 'm/d/Y', 'm/d/y', 'Ymd'] as $fmt) {
        $d = DateTime::createFromFormat($fmt, $str);
        if ($d && $d->format($fmt) === $str) return $d;
    }
    try { return new DateTime($str); } catch (Exception $e) { return null; }
}

function inRange(?DateTime $d, ?DateTime $start, ?DateTime $end): bool {
    if (!$d) return false;
    if ($start && $d < $start) return false;
    if ($end   && $d > $end)   return false;
    return true;
}

// $is_vintage_item = true forces Vintage / Special Interest regardless of keywords
function classifyEntry($need, $notes, $categories, $is_vintage_item = false) {
    if ($is_vintage_item) return 'Vintage / Special Interest';
    $text = strtolower($need . ' ' . $notes);
    foreach ($categories as $cat => $keywords) {
        if (empty($keywords)) continue;
        foreach ($keywords as $kw) {
            if (strpos($text, $kw) !== false) return $cat;
        }
    }
    return 'Unspecified / Just Need';
}

// ---- Read data ----
$wl_data = [];
if (file_exists($waitlist_file)) {
    $fh = fopen($waitlist_file, 'r');
    fgetcsv($fh);
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) < 13) $row = array_pad($row, 15, '');
        $wl_data[] = $row;
    }
    fclose($fh);
}

$vintage_inv_ids = [];
if (file_exists($inventory_file)) {
    $fh = fopen($inventory_file, 'r');
    fgetcsv($fh);
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) >= 12 && trim($row[11]) === 'Vintage') {
            $vintage_inv_ids[trim($row[0])] = true;
        }
    }
    fclose($fh);
}

// ---- Collect all years with pickup data ----
$all_pickup_years = [];
foreach ($wl_data as $row) {
    if (trim($row[11] ?? '') === 'Y' && !empty(trim($row[12] ?? ''))) {
        $d = parseDate($row[12]);
        if ($d) $all_pickup_years[$d->format('Y')] = true;
    }
}
krsort($all_pickup_years);
$all_pickup_years = array_keys($all_pickup_years);

// ---- Resolve period: custom date range takes precedence over year preset ----
$date_from_raw = trim($_GET['date_from'] ?? '');
$date_to_raw   = trim($_GET['date_to']   ?? '');
$selected_year = isset($_GET['year']) ? trim($_GET['year']) : 'all';
$range_start   = null;
$range_end     = null;
$mode          = 'all'; // 'all' | 'year' | 'range'

if ($date_from_raw !== '' || $date_to_raw !== '') {
    $mode = 'range';
    if ($date_from_raw !== '') {
        $rs = parseDate($date_from_raw);
        if ($rs) { $rs->setTime(0, 0, 0); $range_start = $rs; }
    }
    if ($date_to_raw !== '') {
        $re = parseDate($date_to_raw);
        if ($re) { $re->setTime(23, 59, 59); $range_end = $re; }
    }
} elseif ($selected_year !== 'all' && in_array($selected_year, $all_pickup_years)) {
    $mode        = 'year';
    $range_start = new DateTime($selected_year . '-01-01 00:00:00');
    $range_end   = new DateTime($selected_year . '-12-31 23:59:59');
}

// ---- Build month-to-month fulfillment data ----
$fulfill_by_month = [];
foreach ($wl_data as $row) {
    if (trim($row[11] ?? '') !== 'Y' || empty(trim($row[12] ?? ''))) continue;
    $d = parseDate($row[12]);
    if (!$d) continue;
    if ($mode !== 'all' && !inRange($d, $range_start, $range_end)) continue;
    $key = $d->format('Y-m');
    $fulfill_by_month[$key] = ($fulfill_by_month[$key] ?? 0) + 1;
}
ksort($fulfill_by_month);

// Fill month gaps based on mode
if ($mode === 'year') {
    $filled = [];
    for ($m = 1; $m <= 12; $m++) {
        $key = $selected_year . '-' . str_pad($m, 2, '0', STR_PAD_LEFT);
        $filled[$key] = $fulfill_by_month[$key] ?? 0;
    }
    $fulfill_by_month = $filled;
} elseif ($mode === 'range') {
    // Determine span: use explicit bounds when available, otherwise fall back to data bounds
    $span_start = $range_start;
    $span_end   = $range_end;
    if (!empty($fulfill_by_month)) {
        $keys = array_keys($fulfill_by_month);
        if (!$span_start) $span_start = DateTime::createFromFormat('Y-m', reset($keys));
        if (!$span_end)   $span_end   = DateTime::createFromFormat('Y-m', end($keys));
    }
    if ($span_start && $span_end) {
        $cur = clone $span_start;
        $cur->modify('first day of this month');
        $end_mo = clone $span_end;
        $end_mo->modify('first day of this month');
        $filled = [];
        while ($cur <= $end_mo) {
            $k = $cur->format('Y-m');
            $filled[$k] = $fulfill_by_month[$k] ?? 0;
            $cur->modify('+1 month');
        }
        $fulfill_by_month = $filled;
    }
} elseif (!empty($fulfill_by_month)) {
    $keys  = array_keys($fulfill_by_month);
    $cur   = DateTime::createFromFormat('Y-m', reset($keys));
    $endmo = DateTime::createFromFormat('Y-m', end($keys));
    $filled = [];
    while ($cur <= $endmo) {
        $k = $cur->format('Y-m');
        $filled[$k] = $fulfill_by_month[$k] ?? 0;
        $cur->modify('+1 month');
    }
    $fulfill_by_month = $filled;
}

$chart_labels = [];
$chart_data   = [];
foreach ($fulfill_by_month as $ym => $cnt) {
    $d = DateTime::createFromFormat('Y-m', $ym);
    $chart_labels[] = $d ? $d->format('M Y') : $ym;
    $chart_data[]   = $cnt;
}

// ---- Build demographic data ----
$demo_fulfilled = [];
$demo_waiting   = [];
foreach ($demo_categories as $cat => $kws) {
    $demo_fulfilled[$cat] = 0;
    $demo_waiting[$cat]   = 0;
}

foreach ($wl_data as $row) {
    $picked_up = trim($row[11] ?? '');
    $need      = $row[13] ?? '';
    $notes     = $row[14] ?? '';

    if ($mode !== 'all') {
        $date_col = ($picked_up === 'Y') ? ($row[12] ?? '') : ($row[5] ?? '');
        $d = parseDate($date_col);
        if (!inRange($d, $range_start, $range_end)) continue;
    }

    $inv_id          = trim($row[8] ?? '');
    $is_vintage_item = !empty($inv_id) && isset($vintage_inv_ids[$inv_id]);
    $cat = classifyEntry($need, $notes, $demo_categories, $is_vintage_item);
    if ($picked_up === 'Y') {
        $demo_fulfilled[$cat]++;
    } else {
        $demo_waiting[$cat]++;
    }
}

$demo_labels          = array_keys($demo_fulfilled);
$demo_fulfilled_vals  = array_values($demo_fulfilled);
$demo_waiting_vals    = array_values($demo_waiting);
$total_fulfilled      = array_sum($demo_fulfilled_vals);
$total_waiting        = array_sum($demo_waiting_vals);
$total_all            = $total_fulfilled + $total_waiting;
$fulfill_rate         = $total_all > 0 ? round(($total_fulfilled / $total_all) * 100, 1) : 0;

$chart_labels_json        = json_encode($chart_labels);
$chart_data_json          = json_encode($chart_data);
$demo_labels_json         = json_encode($demo_labels);
$demo_fulfilled_vals_json = json_encode($demo_fulfilled_vals);
$demo_waiting_vals_json   = json_encode($demo_waiting_vals);

// Human-readable scope label
if ($mode === 'all') {
    $scope_label = 'All Time';
} elseif ($mode === 'year') {
    $scope_label = $selected_year;
} else {
    $parts = [];
    if ($range_start) $parts[] = $range_start->format('M j, Y');
    if ($range_end)   $parts[] = $range_end->format('M j, Y');
    $scope_label = count($parts) === 2 ? implode(' — ', $parts) : (reset($parts) ?: 'Custom Range');
}
$generated_on = date('F j, Y \a\t g:i a');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports &mdash; <?= htmlspecialchars($scope_label) ?> &mdash; Laptop Ministry</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
    <style>
        /* ---- Screen layout ---- */
        .report-controls {
            display: flex;
            align-items: flex-end;
            gap: 14px;
            flex-wrap: wrap;
            margin: 10px 0 16px;
            padding: 10px;
            background: #e8e8e8;
            border: 2px solid;
            border-color: #808080 #ffffff #ffffff #808080;
        }
        .ctrl-group {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        .ctrl-group label {
            font-weight: bold;
            font-size: 12px;
        }
        .ctrl-group select,
        .ctrl-group input[type="date"] {
            padding: 3px 6px;
            font-size: 13px;
            border: 2px solid;
            border-color: #808080 #ffffff #ffffff #808080;
            background: #fff;
        }
        .ctrl-divider {
            align-self: center;
            font-size: 12px;
            color: #666;
            padding-bottom: 2px;
        }
        .ctrl-actions {
            display: flex;
            gap: 6px;
            align-items: flex-end;
        }
        .print-btn {
            background: #c0c0c0;
            border: 2px solid;
            border-color: #ffffff #808080 #808080 #ffffff;
            padding: 4px 12px;
            font-size: 13px;
            cursor: pointer;
        }
        .print-btn:active {
            border-color: #808080 #ffffff #ffffff #808080;
        }
        .chart-section { margin: 16px 0; }
        .chart-section h3 {
            background: #000080;
            color: #fff;
            padding: 4px 10px;
            margin: 0 0 8px;
            font-size: 14px;
        }
        .chart-wrap {
            background: #fff;
            border: 2px solid;
            border-color: #808080 #ffffff #ffffff #808080;
            padding: 10px;
        }
        .chart-canvas-container { position: relative; width: 100%; }
        .report-summary {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
            margin-bottom: 12px;
        }
        .report-summary .stat-box { min-width: 120px; }
        .chart-legend {
            display: flex;
            gap: 14px;
            flex-wrap: wrap;
            margin-top: 6px;
            font-size: 12px;
        }
        .legend-dot {
            display: inline-block;
            width: 12px; height: 12px;
            border-radius: 2px;
            margin-right: 4px;
            vertical-align: middle;
        }
        .no-data {
            color: #888;
            font-style: italic;
            padding: 18px 0;
            text-align: center;
        }
        /* Print header — hidden on screen */
        .print-header { display: none; }
        /* Chart print images — hidden on screen, shown on print */
        .chart-print-img { display: none; }

        /* ---- Print styles ---- */
        @media print {
            @page {
                size: letter portrait;
                margin: 0.45in 0.5in 0.4in 0.5in;
            }

            /* --- Undo the OS2 theme's height:100vh on .admin-wrapper and
                   overflow-y:auto on .content — these are the root cause of
                   print overlap because they lock the layout to viewport height. --- */
            html, body {
                height: auto !important;
                overflow: visible !important;
                background: #fff !important;
                font-size: 10px !important;
            }
            .admin-wrapper {
                display: block !important;
                height: auto !important;
                min-height: 0 !important;
                overflow: visible !important;
            }
            .content {
                display: block !important;
                flex: none !important;
                height: auto !important;
                min-height: 0 !important;
                overflow: visible !important;
                padding: 0 !important;
                margin: 0 !important;
                width: 100% !important;
            }
            .window {
                display: block !important;
                height: auto !important;
                overflow: visible !important;
                border: none !important;
                box-shadow: none !important;
                padding: 0 !important;
                margin: 0 !important;
            }
            .window-title { display: none !important; }
            .window-content {
                display: block !important;
                height: auto !important;
                overflow: visible !important;
                padding: 0 !important;
            }

            /* Hide chrome */
            .sidebar,
            .report-controls,
            .no-print               { display: none !important; }

            /* Print header */
            .print-header           { display: block !important; border-bottom: 1.5pt solid #000; margin-bottom: 5pt; padding-bottom: 3pt; }
            .print-header h1        { font-size: 13pt; margin: 0 0 1pt; font-weight: bold; }
            .print-header p         { font-size: 8pt; margin: 0; color: #333; }

            /* Summary stats */
            .report-summary         { display: flex !important; flex-wrap: nowrap !important; gap: 4pt !important; margin-bottom: 5pt !important; }
            .stat-box               { flex: 1 !important; padding: 4pt 5pt !important; min-width: 0 !important; border: 1pt solid #aaa !important; }
            .stat-box .stat-value   { font-size: 13pt !important; color: #000 !important; }
            .stat-box .stat-label   { font-size: 7pt !important; }

            /* Chart sections */
            .chart-section          { display: block !important; page-break-inside: avoid !important; break-inside: avoid !important; margin: 4pt 0 !important; }
            .chart-section h3       { background: none !important; color: #000 !important; border-bottom: 1.5pt solid #000 !important; padding: 0 0 2pt !important; font-size: 10pt !important; margin: 0 0 3pt !important; }
            .chart-wrap             { border: none !important; padding: 0 !important; display: block !important; overflow: visible !important; }
            .chart-canvas-container { display: block !important; overflow: visible !important; height: auto !important; }
            /* Canvas is replaced by a PNG snapshot for print */
            canvas                  { display: none !important; }
            .chart-print-img        { display: block !important; width: 100% !important; height: auto !important; max-width: 100% !important; }

            /* Demographic table */
            .data-table             { font-size: 8.5px !important; }
            .data-table th,
            .data-table td          { padding: 2px 3px !important; }
            .progress-bar-track     { height: 5px !important; }
            /* Force background fills to print — browsers suppress bg colors by default */
            .progress-bar-track,
            .progress-bar-fill      {
                -webkit-print-color-adjust: exact !important;
                print-color-adjust: exact !important;
                color-adjust: exact !important;
            }
            .chart-legend           { font-size: 8pt !important; margin-top: 2pt !important; }
            .demo-footnote          { display: none !important; }
        }
    </style>
</head>
<body>
<div class="admin-wrapper">
    <?php include '_sidebar.php'; ?>
    <main class="content">
        <div class="window">
            <div class="window-title">Reports &mdash; <?= htmlspecialchars($scope_label) ?></div>
            <div class="window-content">

                <!-- Print-only header -->
                <div class="print-header">
                    <h1>Laptop Ministry &mdash; Fulfillment Report</h1>
                    <p>Period: <strong><?= htmlspecialchars($scope_label) ?></strong> &nbsp;|&nbsp; Generated: <?= htmlspecialchars($generated_on) ?></p>
                </div>

                <!-- Controls (hidden on print) -->
                <form method="get" class="report-controls no-print">
                    <div class="ctrl-group">
                        <label for="year-select">Preset Period</label>
                        <select id="year-select" name="year">
                            <option value="all" <?= ($mode !== 'range' && $selected_year === 'all') ? 'selected' : '' ?>>&#9733; All Time</option>
                            <?php foreach ($all_pickup_years as $yr): ?>
                                <option value="<?= htmlspecialchars($yr) ?>"
                                    <?= ($mode !== 'range' && $selected_year === $yr) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($yr) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="ctrl-divider">&mdash; or &mdash;</div>

                    <div class="ctrl-group">
                        <label for="date_from">Custom From</label>
                        <input type="date" id="date_from" name="date_from"
                            value="<?= htmlspecialchars($date_from_raw) ?>">
                    </div>
                    <div class="ctrl-group">
                        <label for="date_to">To</label>
                        <input type="date" id="date_to" name="date_to"
                            value="<?= htmlspecialchars($date_to_raw) ?>">
                    </div>

                    <div class="ctrl-actions">
                        <button type="submit">Apply</button>
                        <?php if ($mode === 'range'): ?>
                            <a href="reports.php?year=<?= htmlspecialchars($selected_year) ?>">
                                <button type="button">Clear Range</button>
                            </a>
                        <?php endif; ?>
                        <button type="button" class="print-btn" onclick="printReport()">&#128438; Print</button>
                    </div>
                </form>

                <!-- Summary Stats -->
                <div class="report-summary">
                    <div class="stat-box">
                        <div class="stat-value"><?= $total_all ?></div>
                        <div class="stat-label">Total Entries</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-value"><?= $total_fulfilled ?></div>
                        <div class="stat-label">Fulfilled</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-value"><?= $total_waiting ?></div>
                        <div class="stat-label">Still Waiting</div>
                    </div>
                    <div class="stat-box">
                        <div class="stat-value"><?= $fulfill_rate ?>%</div>
                        <div class="stat-label">Fulfillment Rate</div>
                    </div>
                </div>

                <!-- Fulfillment Rate Chart -->
                <div class="chart-section">
                    <h3>&#128200; Month-to-Month Fulfillment</h3>
                    <div class="chart-wrap">
                        <?php if (empty($chart_data) || array_sum($chart_data) === 0): ?>
                            <p class="no-data">No fulfillment data available for this period.</p>
                        <?php else: ?>
                            <div class="chart-canvas-container" id="fulfill-canvas-wrap" style="height:260px;">
                                <canvas id="fulfillChart"></canvas>
                                <img id="fulfill-print-img" class="chart-print-img" src="" alt="Fulfillment chart">
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Demographic Chart -->
                <div class="chart-section">
                    <h3>&#128101; Need Demographics &mdash; Fulfilled vs. Waiting</h3>
                    <div class="chart-wrap">
                        <?php if ($total_all === 0): ?>
                            <p class="no-data">No data available for this period.</p>
                        <?php else: ?>
                            <div class="chart-canvas-container" id="demo-canvas-wrap" style="height:300px;">
                                <canvas id="demoChart"></canvas>
                                <img id="demo-print-img" class="chart-print-img" src="" alt="Demographics chart">
                            </div>
                            <div class="chart-legend">
                                <span><span class="legend-dot" style="background:#2255aa;"></span>Fulfilled (Picked Up)</span>
                                <span><span class="legend-dot" style="background:#cc6600;"></span>Still Waiting</span>
                            </div>
                            <p class="demo-footnote" style="font-size:11px;color:#555;margin-top:5px;">
                                Classified by keyword match on Need &amp; Notes.
                                <strong>Vintage / Special Interest</strong> also triggers when a Vintage inventory item was provided.
                                First match wins; unmatched entries fall into Unspecified / Just Need.
                            </p>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Demographic Breakdown Table -->
                <?php if ($total_all > 0): ?>
                <div class="chart-section">
                    <h3>&#128203; Demographic Breakdown</h3>
                    <div class="chart-wrap">
                        <table class="data-table" style="table-layout:auto;font-size:12px;">
                            <thead>
                                <tr>
                                    <th>Category</th>
                                    <th style="text-align:center;width:70px;">Fulfilled</th>
                                    <th style="text-align:center;width:70px;">Waiting</th>
                                    <th style="text-align:center;width:55px;">Total</th>
                                    <th>Fulfillment %</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($demo_labels as $i => $cat): ?>
                                <?php
                                    $f   = $demo_fulfilled_vals[$i];
                                    $w   = $demo_waiting_vals[$i];
                                    $t   = $f + $w;
                                    $pct = $t > 0 ? round(($f / $t) * 100, 1) : 0;
                                ?>
                                <tr>
                                    <td><?= htmlspecialchars($cat) ?></td>
                                    <td style="text-align:center;"><?= $f ?></td>
                                    <td style="text-align:center;"><?= $w ?></td>
                                    <td style="text-align:center;"><?= $t ?></td>
                                    <td>
                                        <?php if ($t > 0): ?>
                                        <div style="display:flex;align-items:center;gap:6px;">
                                            <div class="progress-bar-track" style="flex:1;height:10px;margin:0;">
                                                <div class="progress-bar-fill bar-blue" style="width:<?= $pct ?>%;"></div>
                                            </div>
                                            <span style="white-space:nowrap;font-size:11px;"><?= $pct ?>%</span>
                                        </div>
                                        <?php else: ?>&mdash;<?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                                <tr style="font-weight:bold;background:#e8e8e8;">
                                    <td>Total</td>
                                    <td style="text-align:center;"><?= $total_fulfilled ?></td>
                                    <td style="text-align:center;"><?= $total_waiting ?></td>
                                    <td style="text-align:center;"><?= $total_all ?></td>
                                    <td>
                                        <div style="display:flex;align-items:center;gap:6px;">
                                            <div class="progress-bar-track" style="flex:1;height:10px;margin:0;">
                                                <div class="progress-bar-fill bar-blue" style="width:<?= $fulfill_rate ?>%;"></div>
                                            </div>
                                            <span style="white-space:nowrap;font-size:11px;"><?= $fulfill_rate ?>%</span>
                                        </div>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
                <?php endif; ?>

            </div><!-- /window-content -->
        </div>
    </main>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script>
Chart.defaults.font.family = "'MS Sans Serif', 'Segoe UI', Arial, sans-serif";
Chart.defaults.font.size = 11;
Chart.defaults.color = '#333';

var fulfillChartObj = null;
var demoChartObj    = null;

<?php if (!empty($chart_data) && array_sum($chart_data) > 0): ?>
(function() {
    var ctx = document.getElementById('fulfillChart');
    if (!ctx) return;
    fulfillChartObj = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: <?= $chart_labels_json ?>,
            datasets: [{
                label: 'Laptops Fulfilled',
                data: <?= $chart_data_json ?>,
                backgroundColor: 'rgba(34, 85, 170, 0.75)',
                borderColor: '#000080',
                borderWidth: 1,
                borderRadius: 2
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            animation: { duration: 400 },
            plugins: {
                legend: { display: false },
                tooltip: {
                    callbacks: {
                        title: function(items) { return items[0].label; },
                        label: function(item) { return item.raw + ' fulfilled'; }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: { precision: 0 },
                    title: { display: true, text: 'Fulfillments' }
                },
                x: { title: { display: true, text: 'Month' } }
            }
        }
    });
})();
<?php endif; ?>

<?php if ($total_all > 0): ?>
(function() {
    var ctx = document.getElementById('demoChart');
    if (!ctx) return;
    demoChartObj = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: <?= $demo_labels_json ?>,
            datasets: [
                {
                    label: 'Fulfilled',
                    data: <?= $demo_fulfilled_vals_json ?>,
                    backgroundColor: 'rgba(34, 85, 170, 0.8)',
                    borderColor: '#000080',
                    borderWidth: 1,
                    borderRadius: 2
                },
                {
                    label: 'Still Waiting',
                    data: <?= $demo_waiting_vals_json ?>,
                    backgroundColor: 'rgba(204, 102, 0, 0.75)',
                    borderColor: '#993300',
                    borderWidth: 1,
                    borderRadius: 2
                }
            ]
        },
        options: {
            indexAxis: 'y',
            responsive: true,
            maintainAspectRatio: false,
            animation: { duration: 400 },
            plugins: {
                legend: { display: false },
                tooltip: {
                    callbacks: {
                        label: function(item) {
                            return item.dataset.label + ': ' + item.raw;
                        }
                    }
                }
            },
            scales: {
                x: {
                    beginAtZero: true,
                    ticks: { precision: 0 },
                    title: { display: true, text: 'People' }
                },
                y: { ticks: { autoSkip: false } }
            }
        }
    });
})();
<?php endif; ?>

// Snapshot charts to <img> elements before printing.
// Canvas elements are unreliable in print; PNG data-URLs render perfectly.
var CHART_PRINT_MAP = [
    { chartObj: function() { return fulfillChartObj; }, imgId: 'fulfill-print-img' },
    { chartObj: function() { return demoChartObj;    }, imgId: 'demo-print-img'    }
];

function snapshotChartsForPrint() {
    CHART_PRINT_MAP.forEach(function(entry) {
        var ch  = entry.chartObj();
        var img = document.getElementById(entry.imgId);
        if (!ch || !img) return;
        // Capture the current canvas as a PNG data URL
        img.src = ch.canvas.toDataURL('image/png');
    });
}

function clearChartSnapshots() {
    CHART_PRINT_MAP.forEach(function(entry) {
        var img = document.getElementById(entry.imgId);
        if (img) img.src = '';
    });
}

window.addEventListener('beforeprint', snapshotChartsForPrint);
window.addEventListener('afterprint',  clearChartSnapshots);

function printReport() {
    snapshotChartsForPrint();
    // Let the browser paint the img elements before opening the dialog
    setTimeout(function() { window.print(); }, 80);
}

// When a date field is filled, clear the year preset so both don't conflict visually
(function() {
    var fromEl = document.getElementById('date_from');
    var toEl   = document.getElementById('date_to');
    var yrEl   = document.getElementById('year-select');
    if (!fromEl || !toEl || !yrEl) return;
    function onDateChange() {
        if (fromEl.value || toEl.value) yrEl.value = 'all';
    }
    fromEl.addEventListener('change', onDateChange);
    toEl.addEventListener('change', onDateChange);
    yrEl.addEventListener('change', function() {
        if (yrEl.value !== 'all') { fromEl.value = ''; toEl.value = ''; }
    });
})();
</script>
</body>
</html>
