<?php
/**
 * setup.php — Laptop Ministry Accounting System
 *
 * Initial install / admin credential reset utility.
 * Self-deletes on successful completion.
 *
 * RECONFIGURE MODE:
 *   Create  data/setup.key  containing a secret passphrase
 *   before accessing this page. The file is consumed on success.
 *
 * FRESH INSTALL MODE:
 *   Triggered automatically when data/users.json does not exist.
 *   No key file is required.
 *
 * !! DELETE THIS FILE from the server when you are done. !!
 */

session_start();

define('USERS_FILE',    __DIR__ . '/data/users.json');
define('SETTINGS_FILE', __DIR__ . '/data/org_settings.json');
define('WAITLIST_FILE', __DIR__ . '/data/waitlist.csv');
define('INVENTORY_FILE',__DIR__ . '/data/inventory.csv');
define('KEY_FILE',      __DIR__ . '/data/setup.key');
define('DATA_DIR',      __DIR__ . '/data');

// ----------------------------------------------------------------
// Detect installation state
// ----------------------------------------------------------------
$users_raw = null;
if (file_exists(USERS_FILE)) {
    $decoded = json_decode(file_get_contents(USERS_FILE), true);
    if (is_array($decoded) && count($decoded) > 0) {
        $users_raw = $decoded;
    }
}

$fresh    = ($users_raw === null);                        // No users exist yet
$has_key  = file_exists(KEY_FILE) && trim(file_get_contents(KEY_FILE)) !== '';
$locked   = !$fresh && !$has_key;                        // Users exist but no key → deny

// ----------------------------------------------------------------
// POST handler
// ----------------------------------------------------------------
$errors  = [];
$success = false;

if (!$locked && $_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'run_setup') {

    // CSRF-lite check
    if (!isset($_POST['setup_token']) || $_POST['setup_token'] !== ($_SESSION['setup_token'] ?? '')) {
        $errors[] = 'Invalid form token. Please reload the page and try again.';
    }

    // Setup key verification (reconfigure mode only)
    if (!$fresh && empty($errors)) {
        $submitted_key = trim($_POST['setup_key'] ?? '');
        $expected_key  = trim(file_get_contents(KEY_FILE));
        if ($submitted_key === '' || !hash_equals($expected_key, $submitted_key)) {
            $errors[] = 'Incorrect setup key.';
        }
    }

    // Admin username
    $admin_user = trim($_POST['admin_username'] ?? '');
    if ($admin_user === '') {
        $errors[] = 'Admin username is required.';
    } elseif (!preg_match('/^[a-zA-Z0-9_.\-]{3,32}$/', $admin_user)) {
        $errors[] = 'Username may only contain letters, numbers, underscores, hyphens, and dots (3–32 characters).';
    }

    // Admin password
    $admin_pass = $_POST['admin_password'] ?? '';
    $admin_conf = $_POST['admin_confirm']  ?? '';
    if (strlen($admin_pass) < 10) {
        $errors[] = 'Password must be at least 10 characters.';
    }
    if ($admin_pass !== $admin_conf) {
        $errors[] = 'Passwords do not match.';
    }

    if (empty($errors)) {

        // Create data directory if needed
        if (!is_dir(DATA_DIR)) {
            mkdir(DATA_DIR, 0750, true);
        }

        // Build new admin record
        $new_admin = [
            'id'       => time(),
            'username' => $admin_user,
            'hash'     => password_hash($admin_pass, PASSWORD_BCRYPT, ['cost' => 12]),
            'role'     => 'admin',
        ];

        if ($fresh) {
            $users_out = [$new_admin];
        } else {
            // Keep editors and auditors; replace all existing admins
            $non_admins = array_values(array_filter($users_raw, fn($u) => ($u['role'] ?? '') !== 'admin'));
            $users_out  = array_merge([$new_admin], $non_admins);
        }

        file_put_contents(USERS_FILE, json_encode($users_out, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);

        // On fresh install: write org settings and scaffold empty CSV files
        if ($fresh) {
            $org = [
                'org_name'            => trim($_POST['org_name']            ?? 'Laptop Ministry'),
                'drop_address_line1'  => trim($_POST['drop_address_line1']  ?? ''),
                'drop_address_line2'  => trim($_POST['drop_address_line2']  ?? ''),
                'drop_city_state_zip' => trim($_POST['drop_city_state_zip'] ?? ''),
                'drop_hours'          => trim($_POST['drop_hours']           ?? ''),
                'drop_notes'          => trim($_POST['drop_notes']           ?? ''),
                'contact_name'        => trim($_POST['contact_name']         ?? ''),
                'contact_email'       => trim($_POST['contact_email']        ?? ''),
                'contact_phone'       => trim($_POST['contact_phone']        ?? ''),
                'tax_contact_name'    => trim($_POST['tax_contact_name']     ?? ''),
                'tax_contact_email'   => trim($_POST['tax_contact_email']    ?? ''),
                'tax_contact_phone'   => trim($_POST['tax_contact_phone']    ?? ''),
                'tax_id'              => trim($_POST['tax_id']               ?? ''),
            ];
            file_put_contents(SETTINGS_FILE, json_encode($org, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);

            // Create empty waitlist CSV if it doesn't exist
            if (!file_exists(WAITLIST_FILE)) {
                $fh = fopen(WAITLIST_FILE, 'w');
                fputcsv($fh, ['number','first_name','last_name','contact','store_date','request_date',
                              'priority','computer_ready','inventory_id','computer_desc','computer_os',
                              'picked_up','pickup_date','need_desc','notes']);
                fclose($fh);
            }

            // Create empty inventory CSV if it doesn't exist
            if (!file_exists(INVENTORY_FILE)) {
                $fh = fopen(INVENTORY_FILE, 'w');
                fputcsv($fh, ['id','entry_date','date_received','donor_name','donor_contact',
                              'tax_letter','item_value','item_description','discarded','notes',
                              'item_type','item_status']);
                fclose($fh);
            }
        }

        // Remove setup.key
        if (file_exists(KEY_FILE)) {
            unlink(KEY_FILE);
        }

        // Regenerate session token so it can't be replayed
        session_regenerate_id(true);
        $_SESSION['setup_done']  = true;
        $_SESSION['setup_fresh'] = $fresh;

        // Self-delete — PHP keeps the script in memory so redirect still works
        unlink(__FILE__);
        header('Location: index.php?setup=done');
        exit;
    }

    // On validation failure, regenerate token to prevent replay
    $_SESSION['setup_token'] = bin2hex(random_bytes(24));
}

// ----------------------------------------------------------------
// Generate CSRF-lite token for this session
// ----------------------------------------------------------------
if (empty($_SESSION['setup_token'])) {
    $_SESSION['setup_token'] = bin2hex(random_bytes(24));
}
$token = $_SESSION['setup_token'];

// How many non-admin accounts exist (shown in reconfigure mode)
$non_admin_count = 0;
if (!$fresh && is_array($users_raw)) {
    foreach ($users_raw as $u) {
        if (($u['role'] ?? '') !== 'admin') $non_admin_count++;
    }
}
$existing_admin_count = is_array($users_raw) ? count($users_raw) - $non_admin_count : 0;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Setup — Laptop Ministry Accounting</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
    <style>
        .setup-wrapper {
            max-width: 680px;
            margin: 30px auto;
            padding: 0 16px 40px;
        }
        .setup-alert {
            padding: 10px 14px;
            margin-bottom: 14px;
            font-size: 13px;
            border: 2px solid;
        }
        .setup-alert-warn {
            background: #fff8dc;
            border-color: #cc8800;
            color: #5a3a00;
        }
        .setup-alert-danger {
            background: #fff0f0;
            border-color: #cc0000;
            color: #660000;
        }
        .setup-alert-info {
            background: #e8f4fb;
            border-color: #0066aa;
            color: #003355;
        }
        .setup-mode-badge {
            display: inline-block;
            padding: 2px 8px;
            font-size: 11px;
            font-weight: bold;
            letter-spacing: 0.5px;
            text-transform: uppercase;
            border: 1px solid;
            margin-left: 8px;
            vertical-align: middle;
        }
        .badge-fresh   { background: #e8f5e9; border-color: #006600; color: #004400; }
        .badge-reconfig{ background: #fff3cd; border-color: #886600; color: #554400; }
        .badge-locked  { background: #f8d7da; border-color: #990000; color: #660000; }
        .pw-strength {
            height: 4px;
            margin-top: 4px;
            background: #ddd;
            border: 1px solid #bbb;
        }
        .pw-strength-bar {
            height: 100%;
            width: 0;
            transition: width 0.2s, background 0.2s;
        }
        .pw-hint {
            font-size: 11px;
            color: #555;
            margin-top: 3px;
        }
        .setup-section-title {
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            color: #444;
            margin: 16px 0 4px;
            border-bottom: 1px solid #ccc;
            padding-bottom: 3px;
        }
        .locked-box {
            text-align: center;
            padding: 30px 20px;
        }
        .locked-box h2 { margin-bottom: 10px; }
        .locked-box p  { font-size: 13px; color: #444; }
        code {
            background: #f0f0f0;
            border: 1px solid #ccc;
            padding: 1px 5px;
            font-family: monospace;
            font-size: 13px;
        }
    </style>
</head>
<body>
<div class="setup-wrapper">

    <!-- ========== LOCKED OUT ========== -->
    <?php if ($locked): ?>
    <div class="window">
        <div class="window-title">
            Setup &mdash; Access Denied
            <span class="setup-mode-badge badge-locked">Locked</span>
        </div>
        <div class="window-content">
            <div class="locked-box">
                <h2>&#128274; Setup is not available</h2>
                <p>
                    User accounts already exist and no setup key was found.<br>
                    This page cannot be used without first creating a key file.
                </p>
                <p style="margin-top:16px;">
                    To reset admin credentials, create the file:<br>
                    <code>data/setup.key</code><br>
                    containing a secret passphrase, then reload this page.
                </p>
                <p style="margin-top:16px;font-size:12px;color:#666;">
                    If you no longer need this utility, delete
                    <code>setup.php</code> from the server.
                </p>
            </div>
        </div>
    </div>

    <?php else: ?>

    <!-- ========== SECURITY WARNING ========== -->
    <div class="setup-alert setup-alert-warn">
        <strong>&#9888; Security notice:</strong>
        This setup utility grants full administrative access.
        <strong>Delete <code>setup.php</code> from the server immediately after use.</strong>
        It will self-delete on success, but verify it is gone.
    </div>

    <?php if ($fresh): ?>
    <div class="setup-alert setup-alert-info">
        <strong>&#128190; Fresh install detected.</strong>
        No user accounts were found. Complete the form below to initialize the system.
    </div>
    <?php else: ?>
    <div class="setup-alert setup-alert-warn">
        <strong>&#128274; Reconfigure mode.</strong>
        <?= $existing_admin_count ?> existing admin account<?= $existing_admin_count !== 1 ? 's' : '' ?> will be replaced.
        <?= $non_admin_count ?> editor / auditor account<?= $non_admin_count !== 1 ? 's' : '' ?> will be preserved.
    </div>
    <?php endif; ?>

    <!-- ========== ERRORS ========== -->
    <?php if (!empty($errors)): ?>
    <div class="setup-alert setup-alert-danger">
        <strong>&#10005; Please correct the following:</strong>
        <ul style="margin:6px 0 0 18px;padding:0;">
            <?php foreach ($errors as $e): ?>
                <li><?= htmlspecialchars($e) ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
    <?php endif; ?>

    <!-- ========== SETUP FORM ========== -->
    <div class="window">
        <div class="window-title">
            <?= $fresh ? '&#128190; Initial Setup' : '&#9881; Reset Admin Credentials' ?>
            <span class="setup-mode-badge <?= $fresh ? 'badge-fresh' : 'badge-reconfig' ?>">
                <?= $fresh ? 'Fresh Install' : 'Reconfigure' ?>
            </span>
        </div>
        <div class="window-content">
            <form method="post" action="setup.php" autocomplete="off">
                <input type="hidden" name="action"      value="run_setup">
                <input type="hidden" name="setup_token" value="<?= htmlspecialchars($token) ?>">

                <!-- Setup key (reconfigure only) -->
                <?php if (!$fresh): ?>
                <fieldset>
                    <legend>Setup Key</legend>
                    <div class="form-row">
                        <label>Key from <code>data/setup.key</code> *</label>
                        <input type="password" name="setup_key" required
                               autocomplete="new-password" placeholder="Enter the passphrase from setup.key">
                    </div>
                    <p style="font-size:11px;color:#555;margin:4px 0 0;">
                        The key file will be deleted on success.
                    </p>
                </fieldset>
                <br>
                <?php endif; ?>

                <!-- Admin account -->
                <fieldset>
                    <legend>Admin Account</legend>
                    <div class="form-row">
                        <label>Username *</label>
                        <input type="text" name="admin_username" required
                               pattern="[a-zA-Z0-9_.\-]{3,32}"
                               title="3–32 characters: letters, numbers, _ . -"
                               autocomplete="off"
                               value="<?= htmlspecialchars($_POST['admin_username'] ?? '') ?>">
                    </div>
                    <div class="form-inline">
                        <div class="form-row">
                            <label>Password * <span style="font-weight:normal;font-size:11px;">(min 10 characters)</span></label>
                            <input type="password" name="admin_password" id="admin_password" required
                                   minlength="10" autocomplete="new-password"
                                   oninput="updateStrength(this.value)">
                            <div class="pw-strength"><div class="pw-strength-bar" id="pw-bar"></div></div>
                            <div class="pw-hint" id="pw-hint">Enter a password</div>
                        </div>
                        <div class="form-row">
                            <label>Confirm Password *</label>
                            <input type="password" name="admin_confirm" required
                                   minlength="10" autocomplete="new-password"
                                   oninput="checkMatch()">
                            <div class="pw-hint" id="pw-match-hint"></div>
                        </div>
                    </div>
                </fieldset>

                <!-- Org settings (fresh install only) -->
                <?php if ($fresh): ?>
                <br>
                <fieldset>
                    <legend>Organization <span style="font-weight:normal;font-size:11px;">(can be updated later in Dashboard)</span></legend>

                    <div class="form-row">
                        <label>Organization Name</label>
                        <input type="text" name="org_name"
                               value="<?= htmlspecialchars($_POST['org_name'] ?? 'Laptop Ministry') ?>">
                    </div>
                </fieldset>

                <br>
                <fieldset>
                    <legend>Drop-off / Donation Address <span style="font-weight:normal;font-size:11px;">(optional &mdash; shown on public page)</span></legend>
                    <div class="form-inline">
                        <div class="form-row">
                            <label>Address Line 1</label>
                            <input type="text" name="drop_address_line1"
                                   placeholder="Street address"
                                   value="<?= htmlspecialchars($_POST['drop_address_line1'] ?? '') ?>">
                        </div>
                        <div class="form-row">
                            <label>Address Line 2</label>
                            <input type="text" name="drop_address_line2"
                                   placeholder="Suite, room, c/o..."
                                   value="<?= htmlspecialchars($_POST['drop_address_line2'] ?? '') ?>">
                        </div>
                        <div class="form-row">
                            <label>City, State, ZIP</label>
                            <input type="text" name="drop_city_state_zip"
                                   placeholder="Chicago, IL 60601"
                                   value="<?= htmlspecialchars($_POST['drop_city_state_zip'] ?? '') ?>">
                        </div>
                    </div>
                    <div class="form-row">
                        <label>Drop-off Hours</label>
                        <input type="text" name="drop_hours"
                               placeholder="e.g. Mon–Fri 9am–5pm, or By appointment"
                               value="<?= htmlspecialchars($_POST['drop_hours'] ?? '') ?>">
                    </div>
                    <div class="form-row">
                        <label>Additional Notes</label>
                        <textarea name="drop_notes"
                                  placeholder="Parking info, entrance instructions, etc."><?= htmlspecialchars($_POST['drop_notes'] ?? '') ?></textarea>
                    </div>
                </fieldset>

                <br>
                <fieldset>
                    <legend>Donation Contact <span style="font-weight:normal;font-size:11px;">(optional)</span></legend>
                    <div class="form-inline">
                        <div class="form-row">
                            <label>Name / Title</label>
                            <input type="text" name="contact_name"
                                   placeholder="e.g. Donations Coordinator"
                                   value="<?= htmlspecialchars($_POST['contact_name'] ?? '') ?>">
                        </div>
                        <div class="form-row">
                            <label>Email</label>
                            <input type="email" name="contact_email"
                                   value="<?= htmlspecialchars($_POST['contact_email'] ?? '') ?>">
                        </div>
                        <div class="form-row">
                            <label>Phone</label>
                            <input type="tel" name="contact_phone"
                                   value="<?= htmlspecialchars($_POST['contact_phone'] ?? '') ?>">
                        </div>
                    </div>
                </fieldset>

                <br>
                <fieldset>
                    <legend>Tax-Exempt / Receipt Contact <span style="font-weight:normal;font-size:11px;">(optional)</span></legend>
                    <div class="form-inline">
                        <div class="form-row">
                            <label>EIN / Tax-Exempt ID</label>
                            <input type="text" name="tax_id"
                                   placeholder="xx-xxxxxxx"
                                   value="<?= htmlspecialchars($_POST['tax_id'] ?? '') ?>">
                        </div>
                        <div class="form-row">
                            <label>Name / Title</label>
                            <input type="text" name="tax_contact_name"
                                   placeholder="e.g. Church Administrator"
                                   value="<?= htmlspecialchars($_POST['tax_contact_name'] ?? '') ?>">
                        </div>
                        <div class="form-row">
                            <label>Email</label>
                            <input type="email" name="tax_contact_email"
                                   value="<?= htmlspecialchars($_POST['tax_contact_email'] ?? '') ?>">
                        </div>
                        <div class="form-row">
                            <label>Phone</label>
                            <input type="tel" name="tax_contact_phone"
                                   value="<?= htmlspecialchars($_POST['tax_contact_phone'] ?? '') ?>">
                        </div>
                    </div>
                </fieldset>
                <?php endif; ?>

                <div style="margin-top:16px;display:flex;align-items:center;gap:14px;flex-wrap:wrap;">
                    <button type="submit">
                        <?= $fresh ? '&#10003; Initialize System' : '&#9881; Reset Admin &amp; Delete Key' ?>
                    </button>
                    <span style="font-size:11px;color:#666;">
                        This file will self-delete on success.
                    </span>
                </div>
            </form>
        </div>
    </div>

    <?php endif; ?>

    <p style="font-size:11px;color:#888;margin-top:10px;text-align:center;">
        Laptop Ministry Accounting System &mdash; Setup Utility
    </p>
</div>

<script>
function updateStrength(pw) {
    var bar  = document.getElementById('pw-bar');
    var hint = document.getElementById('pw-hint');
    var score = 0;
    if (pw.length >= 10) score++;
    if (pw.length >= 14) score++;
    if (/[A-Z]/.test(pw)) score++;
    if (/[0-9]/.test(pw)) score++;
    if (/[^a-zA-Z0-9]/.test(pw)) score++;

    var pct    = Math.min(100, score * 20);
    var colors = ['#cc0000','#cc5500','#cc8800','#448800','#007700'];
    var labels = ['Very weak','Weak','Fair','Good','Strong'];
    bar.style.width      = pct + '%';
    bar.style.background = colors[score > 0 ? score - 1 : 0];
    hint.textContent     = pw.length === 0 ? 'Enter a password' : labels[score > 0 ? score - 1 : 0];
    hint.style.color     = colors[score > 0 ? score - 1 : 0];
    checkMatch();
}
function checkMatch() {
    var pw   = document.getElementById('admin_password').value;
    var hint = document.getElementById('pw-match-hint');
    var conf = document.querySelector('[name="admin_confirm"]').value;
    if (conf === '') { hint.textContent = ''; return; }
    if (pw === conf) {
        hint.textContent = '✓ Passwords match';
        hint.style.color = '#007700';
    } else {
        hint.textContent = '✗ Does not match';
        hint.style.color = '#cc0000';
    }
}
</script>
</body>
</html>
