<?php
require_once 'auth.php';
require_admin();

$users_file = __DIR__ . '/data/users.json';
$users = json_decode(file_get_contents($users_file), true);
$message = '';
$msg_type = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf();

    if (isset($_POST['add_user'])) {
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        $role = $_POST['role'] ?? 'editor';

        if (empty($username) || empty($password)) {
            $message = 'Username and password are required.';
            $msg_type = 'error';
        } elseif (strlen($password) < 6) {
            $message = 'Password must be at least 6 characters.';
            $msg_type = 'error';
        } else {
            // Check for duplicate username
            $duplicate = false;
            foreach ($users as $u) {
                if (strtolower($u['username']) === strtolower($username)) {
                    $duplicate = true;
                    break;
                }
            }
            if ($duplicate) {
                $message = 'Username already exists.';
                $msg_type = 'error';
            } else {
                $users[] = [
                    'id' => time(),
                    'username' => $username,
                    'hash' => password_hash($password, PASSWORD_DEFAULT),
                    'role' => $role
                ];
                file_put_contents($users_file, json_encode($users, JSON_PRETTY_PRINT), LOCK_EX);
                $message = "User '$username' added.";
                $msg_type = 'success';
            }
        }
    }

    if (isset($_POST['delete_user'])) {
        $user_id = $_POST['user_id'] ?? '';
        if ($user_id == $_SESSION['user_id']) {
            $message = 'You cannot delete yourself.';
            $msg_type = 'error';
        } else {
            $users = array_values(array_filter($users, fn($u) => $u['id'] != $user_id));
            file_put_contents($users_file, json_encode($users, JSON_PRETTY_PRINT), LOCK_EX);
            $message = 'User deleted.';
            $msg_type = 'success';
        }
    }

    // Reload after changes
    $users = json_decode(file_get_contents($users_file), true);
}

$token = csrf_token();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Users - Laptop Ministry Accounting</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
</head>
<body>
    <div class="admin-wrapper">
        <?php include '_sidebar.php'; ?>
        <main class="content">
            <div class="window">
                <div class="window-title">User Management</div>
                <div class="window-content">
                    <?php if ($message): ?>
                        <p class="<?= $msg_type ?>"><?= htmlspecialchars($message) ?></p>
                    <?php endif; ?>

                    <fieldset>
                        <legend>Add New User</legend>
                        <form method="POST">
                            <input type="hidden" name="csrf_token" value="<?= $token ?>">
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>Username:</label>
                                    <input type="text" name="username" required>
                                </div>
                                <div class="form-row">
                                    <label>Password:</label>
                                    <input type="password" name="password" required>
                                </div>
                                <div class="form-row">
                                    <label>Role:</label>
                                    <select name="role">
                                        <option value="editor">Editor</option>
                                        <option value="auditor">Auditor (read-only)</option>
                                        <option value="admin">Admin</option>
                                    </select>
                                </div>
                            </div>
                            <button type="submit" name="add_user">Add User</button>
                        </form>
                    </fieldset>

                    <br>

                    <fieldset>
                        <legend>Existing Users</legend>
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Username</th>
                                    <th>Role</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($users as $user): ?>
                                <tr>
                                    <td><?= htmlspecialchars($user['id']) ?></td>
                                    <td><?= htmlspecialchars($user['username']) ?></td>
                                    <td><?= htmlspecialchars($user['role']) ?></td>
                                    <td>
                                        <?php if ($user['id'] == $_SESSION['user_id']): ?>
                                            (Current User)
                                        <?php else: ?>
                                            <form method="POST" style="margin:0;" onsubmit="return confirm('Delete user <?= htmlspecialchars($user['username']) ?>?');">
                                                <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                                <input type="hidden" name="user_id" value="<?= htmlspecialchars($user['id']) ?>">
                                                <button type="submit" name="delete_user" class="btn-danger">Delete</button>
                                            </form>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </fieldset>
                </div>
            </div>
        </main>
    </div>
</body>
</html>
