<?php
require_once 'auth.php';

$csv_file = __DIR__ . '/data/waitlist.csv';
$inventory_file = __DIR__ . '/data/inventory.csv';
$headers_list = ['number','first_name','last_name','contact','store_date','request_date','priority','computer_ready','inventory_id','computer_desc','computer_os','picked_up','pickup_date','need_desc','notes'];

function readWaitlist($file) {
    $data = [];
    if (!file_exists($file)) return $data;
    $fh = fopen($file, 'r');
    $headers = fgetcsv($fh);
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) >= 15) {
            $data[] = $row;
        } elseif (count($row) > 0) {
            $data[] = array_pad($row, 15, '');
        }
    }
    fclose($fh);
    return $data;
}

function writeWaitlist($file, $data) {
    global $headers_list;
    $fh = fopen($file, 'w');
    if (flock($fh, LOCK_EX)) {
        fputcsv($fh, $headers_list);
        foreach ($data as $row) {
            fputcsv($fh, $row);
        }
        flock($fh, LOCK_UN);
    }
    fclose($fh);
}

function getNextNumber($data) {
    $max = 0;
    foreach ($data as $row) {
        $n = intval($row[0]);
        if ($n > $max) $max = $n;
    }
    return $max + 1;
}

// Load inventory for the assignment picker
function loadInventoryItems($file) {
    $items = [];
    if (!file_exists($file)) return $items;
    $fh = fopen($file, 'r');
    fgetcsv($fh); // skip headers
    while (($row = fgetcsv($fh)) !== false) {
        if (count($row) >= 10) {
            $item_type   = $row[10] ?? '';
            $item_status = $row[11] ?? '';
            if ($item_type === 'Part') continue;                               // accessory parts
            if ($item_status === 'Repair' || $item_status === 'Parts') continue; // not fulfillable
            $items[] = ['id' => $row[0], 'desc' => $row[7], 'discarded' => $row[8]];
        }
    }
    fclose($fh);
    return $items;
}

$message = '';
$msg_type = '';

// Handle POST actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    require_editor();
    verify_csrf();
    $data = readWaitlist($csv_file);

    if ($_POST['action'] === 'add') {
        $new = [
            getNextNumber($data),
            trim($_POST['first_name'] ?? ''),
            trim($_POST['last_name'] ?? ''),
            trim($_POST['contact'] ?? ''),
            date('Ymd'),
            trim($_POST['request_date'] ?? ''),
            $_POST['priority'] ?? 'N',
            $_POST['computer_ready'] ?? 'N',
            trim($_POST['inventory_id'] ?? ''),
            trim($_POST['computer_desc'] ?? ''),
            trim($_POST['computer_os'] ?? ''),
            $_POST['picked_up'] ?? 'N',
            trim($_POST['pickup_date'] ?? ''),
            trim($_POST['need_desc'] ?? ''),
            trim($_POST['notes'] ?? '')
        ];
        $data[] = $new;
        writeWaitlist($csv_file, $data);
        header('Location: waitlist.php?msg=added');
        exit;
    }

    if ($_POST['action'] === 'edit' && isset($_POST['edit_number'])) {
        $num = $_POST['edit_number'];
        foreach ($data as &$row) {
            if ($row[0] == $num) {
                $row[1] = trim($_POST['first_name'] ?? '');
                $row[2] = trim($_POST['last_name'] ?? '');
                $row[3] = trim($_POST['contact'] ?? '');
                // row[4] store_date unchanged
                $row[5] = trim($_POST['request_date'] ?? '');
                $row[6] = $_POST['priority'] ?? 'N';
                $row[7] = $_POST['computer_ready'] ?? 'N';
                $row[8] = trim($_POST['inventory_id'] ?? '');
                $row[9] = trim($_POST['computer_desc'] ?? '');
                $row[10] = trim($_POST['computer_os'] ?? '');
                $row[11] = $_POST['picked_up'] ?? 'N';
                $row[12] = trim($_POST['pickup_date'] ?? '');
                $row[13] = trim($_POST['need_desc'] ?? '');
                $row[14] = trim($_POST['notes'] ?? '');
                break;
            }
        }
        unset($row);
        writeWaitlist($csv_file, $data);
        header('Location: waitlist.php?msg=updated');
        exit;
    }

    if ($_POST['action'] === 'delete' && isset($_POST['delete_number'])) {
        $num = $_POST['delete_number'];
        $data = array_filter($data, function($row) use ($num) {
            return $row[0] != $num;
        });
        writeWaitlist($csv_file, array_values($data));
        header('Location: waitlist.php?msg=deleted');
        exit;
    }
}

// Flash messages
if (isset($_GET['msg'])) {
    $msgs = ['added' => 'Entry added.', 'updated' => 'Entry updated.', 'deleted' => 'Entry deleted.'];
    $message = $msgs[$_GET['msg']] ?? '';
    $msg_type = 'success';
}

$data = readWaitlist($csv_file);

// Build available inventory items (not discarded, not already assigned)
$all_inv = loadInventoryItems($inventory_file);
$assigned_ids = [];
foreach ($data as $row) {
    $inv_id = trim($row[8] ?? '');
    if (!empty($inv_id)) $assigned_ids[] = $inv_id;
}
$available_inv = array_filter($all_inv, function($item) use ($assigned_ids) {
    return $item['discarded'] !== 'Yes' && !in_array($item['id'], $assigned_ids);
});
// JSON for JS auto-fill
$inv_json = json_encode(array_values($available_inv));

// Find edit entry
$edit_entry = null;
if (isset($_GET['edit'])) {
    foreach ($data as $row) {
        if ($row[0] == $_GET['edit']) {
            $edit_entry = $row;
            break;
        }
    }
}

$token = csrf_token();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Waitlist - Laptop Ministry Accounting</title>
    <link rel="stylesheet" href="assets/os2-theme.css">
    <link rel="stylesheet" href="assets/custom.css">
</head>
<body>
    <div class="admin-wrapper">
        <?php include '_sidebar.php'; ?>
        <main class="content">
            <div class="window">
                <div class="window-title">Waitlist Management</div>
                <div class="window-content">

                    <?php if ($message): ?>
                        <p class="<?= $msg_type ?>"><?= htmlspecialchars($message) ?></p>
                    <?php endif; ?>

                    <?php if ($edit_entry && can_edit()): ?>
                    <!-- Edit Modal -->
                    <div class="modal-overlay" id="edit-modal">
                        <div class="modal-content">
                            <div class="window-title">Edit Entry #<?= htmlspecialchars($edit_entry[0]) ?></div>
                            <form method="post">
                                <input type="hidden" name="action" value="edit">
                                <input type="hidden" name="edit_number" value="<?= htmlspecialchars($edit_entry[0]) ?>">
                                <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                <div class="form-row">
                                    <label>First Name:</label>
                                    <input type="text" name="first_name" value="<?= htmlspecialchars($edit_entry[1]) ?>" required>
                                </div>
                                <div class="form-row">
                                    <label>Last Name:</label>
                                    <input type="text" name="last_name" value="<?= htmlspecialchars($edit_entry[2]) ?>" required>
                                </div>
                                <div class="form-row">
                                    <label>Contact:</label>
                                    <input type="text" name="contact" value="<?= htmlspecialchars($edit_entry[3]) ?>" required>
                                </div>
                                <div class="form-row">
                                    <label>Request Date:</label>
                                    <input type="date" name="request_date" value="<?= htmlspecialchars($edit_entry[5]) ?>" required>
                                </div>
                                <div class="form-row">
                                    <label>Priority:</label>
                                    <select name="priority">
                                        <option value="N" <?= $edit_entry[6]==='N'?'selected':'' ?>>No</option>
                                        <option value="Y" <?= $edit_entry[6]==='Y'?'selected':'' ?>>Yes</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Computer Ready:</label>
                                    <select name="computer_ready">
                                        <option value="N" <?= $edit_entry[7]==='N'?'selected':'' ?>>No</option>
                                        <option value="Y" <?= $edit_entry[7]==='Y'?'selected':'' ?>>Yes</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Assign Inventory Item:</label>
                                    <select id="edit-inv-picker" onchange="fillEditInv(this)">
                                        <option value="">-- Select from inventory --</option>
                                        <?php foreach ($available_inv as $item): ?>
                                            <option value="<?= htmlspecialchars($item['id']) ?>" data-desc="<?= htmlspecialchars($item['desc']) ?>"><?= htmlspecialchars($item['id']) ?> &mdash; <?= htmlspecialchars($item['desc']) ?></option>
                                        <?php endforeach; ?>
                                        <?php if (!empty($edit_entry[8])): ?>
                                            <option value="<?= htmlspecialchars($edit_entry[8]) ?>" selected data-desc="<?= htmlspecialchars($edit_entry[9]) ?>"><?= htmlspecialchars($edit_entry[8]) ?> &mdash; <?= htmlspecialchars($edit_entry[9]) ?> (current)</option>
                                        <?php endif; ?>
                                        <option value="_manual">Manual entry...</option>
                                    </select>
                                </div>
                                <div id="edit-inv-fields">
                                    <div class="form-row">
                                        <label>Inventory ID#:</label>
                                        <input type="text" name="inventory_id" id="edit-inv-id" value="<?= htmlspecialchars($edit_entry[8]) ?>">
                                    </div>
                                    <div class="form-row">
                                        <label>Computer Description:</label>
                                        <input type="text" name="computer_desc" id="edit-inv-desc" value="<?= htmlspecialchars($edit_entry[9]) ?>">
                                    </div>
                                    <div class="form-row">
                                        <label>Computer OS:</label>
                                        <input type="text" name="computer_os" id="edit-inv-os" value="<?= htmlspecialchars($edit_entry[10]) ?>">
                                    </div>
                                </div>
                                <div class="form-row">
                                    <label>Picked Up:</label>
                                    <select name="picked_up">
                                        <option value="N" <?= $edit_entry[11]==='N'?'selected':'' ?>>No</option>
                                        <option value="Y" <?= $edit_entry[11]==='Y'?'selected':'' ?>>Yes</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Pickup Date:</label>
                                    <input type="date" name="pickup_date" value="<?= htmlspecialchars($edit_entry[12]) ?>">
                                </div>
                                <div class="form-row">
                                    <label>Need Description:</label>
                                    <textarea name="need_desc"><?= htmlspecialchars($edit_entry[13]) ?></textarea>
                                </div>
                                <div class="form-row">
                                    <label>Notes:</label>
                                    <textarea name="notes"><?= htmlspecialchars($edit_entry[14]) ?></textarea>
                                </div>
                                <button type="submit">Save Changes</button>
                                <a href="waitlist.php"><button type="button">Cancel</button></a>
                            </form>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if (can_edit()): ?>
                    <!-- Add New Entry -->
                    <fieldset>
                        <legend>Add New Entry</legend>
                        <form method="post">
                            <input type="hidden" name="action" value="add">
                            <input type="hidden" name="csrf_token" value="<?= $token ?>">
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>First Name:</label>
                                    <input type="text" name="first_name" required>
                                </div>
                                <div class="form-row">
                                    <label>Last Name:</label>
                                    <input type="text" name="last_name" required>
                                </div>
                                <div class="form-row">
                                    <label>Contact:</label>
                                    <input type="text" name="contact" required>
                                </div>
                                <div class="form-row">
                                    <label>Request Date:</label>
                                    <input type="date" name="request_date" required>
                                </div>
                            </div>
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>Priority:</label>
                                    <select name="priority">
                                        <option value="N">No</option>
                                        <option value="Y">Yes</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Computer Ready:</label>
                                    <select name="computer_ready">
                                        <option value="N">No</option>
                                        <option value="Y">Yes</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Assign Inventory:</label>
                                    <select id="add-inv-picker" onchange="fillAddInv(this)">
                                        <option value="">-- Select from inventory --</option>
                                        <?php foreach ($available_inv as $item): ?>
                                            <option value="<?= htmlspecialchars($item['id']) ?>" data-desc="<?= htmlspecialchars($item['desc']) ?>"><?= htmlspecialchars($item['id']) ?> &mdash; <?= htmlspecialchars($item['desc']) ?></option>
                                        <?php endforeach; ?>
                                        <option value="_manual">Manual entry...</option>
                                    </select>
                                </div>
                                <div id="add-inv-fields">
                                    <div class="form-row">
                                        <label>Inventory ID#:</label>
                                        <input type="text" name="inventory_id" id="add-inv-id">
                                    </div>
                                    <div class="form-row">
                                        <label>Computer Desc:</label>
                                        <input type="text" name="computer_desc" id="add-inv-desc">
                                    </div>
                                    <div class="form-row">
                                        <label>Computer OS:</label>
                                        <input type="text" name="computer_os" id="add-inv-os">
                                    </div>
                                </div>
                            </div>
                            <div class="form-inline">
                                <div class="form-row">
                                    <label>Picked Up:</label>
                                    <select name="picked_up">
                                        <option value="N">No</option>
                                        <option value="Y">Yes</option>
                                    </select>
                                </div>
                                <div class="form-row">
                                    <label>Pickup Date:</label>
                                    <input type="date" name="pickup_date">
                                </div>
                            </div>
                            <div class="form-row">
                                <label>Need Description:</label>
                                <textarea name="need_desc"></textarea>
                            </div>
                            <div class="form-row">
                                <label>Notes:</label>
                                <textarea name="notes"></textarea>
                            </div>
                            <button type="submit">Add Entry</button>
                        </form>
                    </fieldset>
                    <?php else: ?>
                    <p><em>Read-only access &mdash; auditor view.</em></p>
                    <?php endif; ?>

                    <!-- Waitlist Table -->
                    <h3>Current Waitlist (<span id="wl-count-total"><?= count($data) ?></span> entries)</h3>
                    <div class="table-search">
                        <label for="wl-search">Search:</label>
                        <input type="search" id="wl-search" placeholder="Filter by any field..." oninput="filterTable('wl-table', this.value, 'wl-shown')">
                        <span id="wl-shown" class="search-count"></span>
                    </div>
                    <div class="table-scroll">
                        <table class="data-table wl-table" id="wl-table">
                            <colgroup>
                                <col class="col-num"><col class="col-name"><col class="col-name">
                                <col class="col-contact"><col class="col-date"><col class="col-date">
                                <col class="col-yn"><col class="col-yn"><col class="col-invid">
                                <col class="col-desc"><col class="col-os"><col class="col-yn">
                                <col class="col-date"><col class="col-need"><col class="col-notes">
                                <?php if (can_edit()): ?><col class="col-actions"><?php endif; ?>
                            </colgroup>
                            <thead>
                                <tr>
                                    <th class="sortable" onclick="sortTable('wl-table',0)">#</th>
                                    <th class="sortable" onclick="sortTable('wl-table',1)">First</th>
                                    <th class="sortable" onclick="sortTable('wl-table',2)">Last</th>
                                    <th class="sortable" onclick="sortTable('wl-table',3)">Contact</th>
                                    <th class="sortable" onclick="sortTable('wl-table',4)">Stored</th>
                                    <th class="sortable" onclick="sortTable('wl-table',5)">Requested</th>
                                    <th class="sortable" onclick="sortTable('wl-table',6)">Prio</th>
                                    <th class="sortable" onclick="sortTable('wl-table',7)">Ready</th>
                                    <th class="sortable" onclick="sortTable('wl-table',8)">Inv ID</th>
                                    <th class="sortable" onclick="sortTable('wl-table',9)">Desc</th>
                                    <th class="sortable" onclick="sortTable('wl-table',10)">OS</th>
                                    <th class="sortable" onclick="sortTable('wl-table',11)">Picked Up</th>
                                    <th class="sortable" onclick="sortTable('wl-table',12)">Pickup Date</th>
                                    <th class="sortable" onclick="sortTable('wl-table',13)">Need</th>
                                    <th class="sortable" onclick="sortTable('wl-table',14)">Notes</th>
                                    <?php if (can_edit()): ?><th>Actions</th><?php endif; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($data as $row): ?>
                                <tr>
                                    <?php for ($i = 0; $i < 15; $i++): ?>
                                        <td><?= htmlspecialchars($row[$i] ?? '') ?></td>
                                    <?php endfor; ?>
                                    <?php if (can_edit()): ?>
                                    <td class="action-btns">
                                        <a href="?edit=<?= htmlspecialchars($row[0]) ?>"><button type="button">Edit</button></a>
                                        <form method="post" onsubmit="return confirm('Delete entry #<?= htmlspecialchars($row[0]) ?>?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="delete_number" value="<?= htmlspecialchars($row[0]) ?>">
                                            <input type="hidden" name="csrf_token" value="<?= $token ?>">
                                            <button type="submit" class="btn-danger">Del</button>
                                        </form>
                                    </td>
                                    <?php endif; ?>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                </div>
            </div>
        </main>
    </div>
<script>
/* ---- Shared table search & sort utilities ---- */
var _sortState = {};

function filterTable(tableId, term, countId) {
    var tbody = document.querySelector('#' + tableId + ' tbody');
    if (!tbody) return;
    var rows = tbody.querySelectorAll('tr');
    var lower = term.toLowerCase().trim();
    var visible = 0;
    rows.forEach(function(row) {
        var show = !lower || row.textContent.toLowerCase().includes(lower);
        row.style.display = show ? '' : 'none';
        if (show) visible++;
    });
    if (countId) {
        var el = document.getElementById(countId);
        if (el) el.textContent = lower ? '(' + visible + ' shown)' : '';
    }
}

function sortTable(tableId, colIdx) {
    var table = document.getElementById(tableId);
    if (!table) return;
    var tbody = table.querySelector('tbody');
    var rows = Array.from(tbody.querySelectorAll('tr'));
    if (!_sortState[tableId]) _sortState[tableId] = {col: -1, dir: 1};
    var state = _sortState[tableId];
    state.dir = (state.col === colIdx) ? state.dir * -1 : 1;
    state.col = colIdx;

    rows.sort(function(a, b) {
        var aVal = a.cells[colIdx] ? a.cells[colIdx].textContent.trim() : '';
        var bVal = b.cells[colIdx] ? b.cells[colIdx].textContent.trim() : '';
        var aNum = parseFloat(aVal.replace(/[$,]/g, ''));
        var bNum = parseFloat(bVal.replace(/[$,]/g, ''));
        if (!isNaN(aNum) && !isNaN(bNum)) return (aNum - bNum) * state.dir;
        return aVal.localeCompare(bVal, undefined, {numeric: true}) * state.dir;
    });
    rows.forEach(function(r) { tbody.appendChild(r); });

    table.querySelectorAll('thead th').forEach(function(th, i) {
        th.classList.remove('sort-asc', 'sort-desc');
        if (i === colIdx) th.classList.add(state.dir === 1 ? 'sort-asc' : 'sort-desc');
    });
}

function fillAddInv(sel) {
    var fields = document.getElementById('add-inv-fields');
    var idInput = document.getElementById('add-inv-id');
    var descInput = document.getElementById('add-inv-desc');
    if (sel.value === '_manual') {
        fields.style.display = 'block';
        idInput.value = '';
        descInput.value = '';
        idInput.focus();
    } else if (sel.value === '') {
        fields.style.display = 'none';
        idInput.value = '';
        descInput.value = '';
    } else {
        fields.style.display = 'block';
        idInput.value = sel.value;
        descInput.value = sel.options[sel.selectedIndex].dataset.desc || '';
    }
}
function fillEditInv(sel) {
    var idInput = document.getElementById('edit-inv-id');
    var descInput = document.getElementById('edit-inv-desc');
    if (sel.value === '_manual') {
        idInput.value = '';
        descInput.value = '';
        idInput.focus();
    } else if (sel.value !== '') {
        idInput.value = sel.value;
        descInput.value = sel.options[sel.selectedIndex].dataset.desc || '';
    }
}
</script>
</body>
</html>
